# Librerías estándar
import base64
import cv2
import io
import logging
import numpy as np
import random
import string
import requests
from collections import Counter, defaultdict
from datetime import datetime, timedelta
import json
import hashlib

# Librerías de Django
from django.contrib import messages
from django.contrib.auth import get_user_model, logout
from django.contrib.auth.decorators import login_required
from django.core.mail import send_mail
from django.db.models import Avg, Count, Sum
from django.db.models.functions import TruncMonth
from django.http import JsonResponse
from django.shortcuts import get_object_or_404, redirect, render
from django.template.loader import render_to_string
from django.utils import timezone
from django.utils.html import strip_tags
from django.utils.translation import gettext as _
from django.views import View
from django.views.decorators.csrf import csrf_exempt
from django.conf import settings
from agroapp.common.decorators import user_required
from django.contrib.contenttypes.models import ContentType

# Librerías de terceros
from PIL import Image  # Para manejar imágenes

# Librerias de Mailchimp
from mailchimp_marketing import Client
import mailchimp_marketing as MailchimpMarketing
from mailchimp_marketing.api_client import ApiClientError

# Importaciones locales
from .forms import (CropForm, CropObservationForm, FieldForm,
                    ScoutProfileForm, ScoutRegistrationForm)
from .models import Crop, CropObservation, Field, Recommendation, Scout, SoilTest, Token
from .utils import generate_recommendations, get_current_scout
from social.models import News, Post
from farmer.models import Farmer
from agronomist.models import Agronomist

from django.shortcuts import render, redirect
from django.contrib.contenttypes.models import ContentType
from django.utils.translation import gettext as _
from django.utils.timezone import now
from django.core.exceptions import ObjectDoesNotExist
import logging


# Logger
User = get_user_model()
logger = logging.getLogger(__name__)


def scout_list(request):
    scouts = Scout.objects.all()
    return render(request, 'scouts/scout_list.html', {'scouts': scouts})

# Función para generar un token de 6 dígitos


def generate_token():
    return ''.join(random.choices(string.digits, k=6))


# Función para enviar el correo con el token
def send_verification_email(scout, token_value):
    # Renderizar el HTML desde la plantilla
    html_message = render_to_string(
        'email/verification_token.html', {'token': token_value})
    # Convertir a texto plano para fallback
    plain_message = strip_tags(html_message)
    subject = _('Tu código de verificación')
    from_email = 'AgriCarm App<no-reply@agricarm.com>'
    recipient_list = [scout.email]

    send_mail(subject, plain_message, from_email,
              recipient_list, html_message=html_message)


# Función de registro adaptada
def register(request):
    if request.method == 'POST':
        form = ScoutRegistrationForm(request.POST)
        if form.is_valid():
            scout = form.save(commit=False)
            scout.save()

            # Generar y guardar el token
            token_value = generate_token()
            token = Token.objects.create(scout=scout, token=token_value)

            # Enviar el correo con el token
            send_verification_email(scout, token_value)

            request.session['email'] = scout.email
            # Redirige a la página de verificación de código
            return redirect('verify')
    else:
        form = ScoutRegistrationForm()
    return render(request, 'registro.html', {'form': form})


# Función de login adaptada para múltiples tipos de usuarios
def login_request(request):
    if request.method == 'POST':
        email = request.POST['email']
        try:
            # Buscar al usuario en los tres modelos posibles
            user = None
            user_model = None

            if Scout.objects.filter(email=email).exists():
                user = Scout.objects.get(email=email)
                user_model = Scout
            elif Farmer.objects.filter(email=email).exists():
                user = Farmer.objects.get(email=email)
                user_model = Farmer
            elif Agronomist.objects.filter(email=email).exists():
                user = Agronomist.objects.get(email=email)
                user_model = Agronomist

            if not user:
                return render(request, 'login.html', {'error': _('Correo electrónico no encontrado')})

            # Generar el token
            token_value = generate_token()

            # Crear un token vinculado genéricamente al usuario
            content_type = ContentType.objects.get_for_model(user_model)
            Token.objects.create(content_type=content_type,
                                 object_id=user.id, token=token_value)

            # Enviar el correo con el token
            send_verification_email(user, token_value)

            # Guardar el email en la sesión para usarlo en la verificación
            request.session['email'] = email
            # Redirige a la página de verificación de código
            return redirect('verify')

        except Exception as e:
            return render(request, 'login.html', {'error': _('Ocurrió un error: ') + str(e)})

    return render(request, 'login.html')


def verify_token(request):
    if request.method == 'POST':
        email = request.session.get('email')

        # Recolecta los dígitos individuales del formulario
        token_value = ''.join([
            request.POST.get(f'digit{i}', '') for i in range(1, 7)
        ])

        logger.info(
            f'Trying to verify token for email: {email}, token_value: {token_value}')

        try:
            # Lista de modelos y dashboards correspondientes
            user_types = [
                (Scout, 'dashboard'),
                (Farmer, 'farmer_dashboard'),
                (Agronomist, 'admin_dashboard'),
            ]

            for model, dashboard in user_types:
                # Buscar el usuario en el modelo actual
                user = model.objects.filter(email=email).first()
                if user:
                    # Buscar el token asociado al usuario
                    token = Token.objects.filter(
                        token=token_value,
                        content_type=ContentType.objects.get_for_model(model),
                        object_id=user.id
                    ).order_by('-created_at').first()

                    if token and token.is_valid():
                        # Configurar la sesión según el modelo
                        request.session[f'{model.__name__.lower()}_id'] = user.id
                        logger.info(
                            f'Token verified successfully for {model.__name__.lower()}_id: {user.id}')
                        return redirect(dashboard)

            # Si no se encuentra un token válido
            logger.warning('Invalid or expired token')
            return render(request, 'verificar.html', {'error': _('Token inválido o expirado')})

        except Exception as e:
            logger.error(f'An unexpected error occurred: {e}')
            return render(request, 'verificar.html', {'error': _('Ocurrió un error inesperado')})

    logger.info('GET request for token verification')
    return render(request, 'verificar.html')


def logout_view(request):
    logout(request)  # Esta función elimina toda la información de la sesión
    # Redirige a la página de login después del logout
    return redirect('login')


def get_current_scout(request):
    scout_id = request.session.get('scout_id')
    if scout_id:
        try:
            return Scout.objects.get(id=scout_id)
        except Scout.DoesNotExist:
            return None
    return None


@user_required('scout')
def fieldlistcreate(request):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    if request.method == 'POST':
        form = FieldForm(request.POST)
        if form.is_valid():
            field = form.save(commit=False)
            field.scout = scout_id  # Asigna el scout actual al nuevo Field
            field.save()
            # Redirige a la misma página después de crear un Field
            return redirect('field-list')
    else:
        form = FieldForm()

    # Filtra los Field por el Scout actual
    fields = Field.objects.filter(scout=scout_id)

    # Preparar datos para el gráfico
    fields_data = [{
        'name': field.name,
        'area': field.size
    } for field in fields]

    has_crops = Crop.objects.filter(field__scout=scout_id).exists()

    return render(request, 'field_list.html', {'fields': fields, 'form': form, 'fields_data': fields_data, 'has_crops': has_crops})


@user_required('scout')
def edit_field(request, field_id):
    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    field = get_object_or_404(Field, id=field_id, scout_id=scout_id)
    # Depuración
    print(
        f"Datos del campo: {field.latitude}, {field.longitude}, {field.size}")

    if request.method == 'POST':
        form = FieldForm(request.POST, instance=field)
        if form.is_valid():
            form.save()
            return redirect('field-list')  # Redirige a la lista de campos
    else:
        form = FieldForm(instance=field)

    return render(request, 'field_list.html', {'form': form, 'field': field})


@user_required('scout')
def delete_field(request, field_id):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    field = get_object_or_404(Field, id=field_id, scout=scout_id)

    if request.method == 'POST':
        field.delete()
        # Redirige a la lista de campos después de la eliminación
        return redirect('field-list')

    return render(request, 'field_list.html', {'field': field})


@user_required('scout')
def crop_list(request):
    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    # Filtrar cultivos y campos relacionados al Scout actual
    fields = Field.objects.filter(scout_id=scout_id)
    crops = Crop.objects.filter(field__scout=scout_id)
    has_crops = crops.exists()
    has_fields = fields.exists()

    return render(request, 'crop_list.html', {
        'crops': crops,
        'scout': scout_id,
        'has_crops': has_crops,
        'has_fields': has_fields,
        'fields': fields,
    })


@user_required('scout')
def crop_type_distribution(request):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    # Filtrar los cultivos del scout autenticado
    crops = Crop.objects.filter(field__scout=scout_id)

    # Verificar si el scout tiene cultivos asociados
    if not crops.exists():
        return JsonResponse({
            'error': _('No hay cultivos asociados a este scout.')
        }, status=404)

    # Obtener el conteo de cultivos por tipo
    crop_types_count = crops.values(
        'crop_type').annotate(count=Count('crop_type'))

    # Verificar si se encontraron tipos de cultivos
    if not crop_types_count:
        return JsonResponse({
            'error': _('No hay datos de tipos de cultivos.')
        }, status=404)

    # Mapear los nombres amigables de los tipos de cultivos
    crop_types_labels = [dict(Crop.CROP_TYPE_CHOICES).get(
        ctype['crop_type'], _('Desconocido')) for ctype in crop_types_count]
    crop_types_data = [ctype['count'] for ctype in crop_types_count]

    # Verificar si se recuperaron datos
    if not crop_types_labels or not crop_types_data:
        return JsonResponse({
            'error': _('No se encontraron datos para los tipos de cultivos.')
        }, status=404)

    # Devolver la información en formato JSON
    return JsonResponse({
        'labels': crop_types_labels,
        'data': crop_types_data,
    })


@user_required('scout')
def edit_crop(request, crop_id):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    crop = get_object_or_404(Crop, id=crop_id, field__scout=scout_id)

    if request.method == 'POST':
        form = CropForm(request.POST, instance=crop)
        if form.is_valid():
            form.save()
            # Redirige a la lista de cultivos después de editar
            return redirect('crop-list')
    else:
        form = CropForm(instance=crop)

    return render(request, 'edit_crop.html', {'form': form, 'crop': crop})


@user_required('scout')
def delete_crop(request, crop_id):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    crop = get_object_or_404(Crop, id=crop_id, field__scout=scout_id)

    if request.method == 'GET':
        crop.delete()  # Eliminar el cultivo
        # Redirige a la lista de cultivos después de eliminar
        return redirect('crop-list')

    # Para cualquier otra solicitud, redirigir a la lista de cultivos
    return redirect('crop-list')


@user_required('scout')
def field_detail(request, field_id):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    field = get_object_or_404(Field, id=field_id)
    # Usar el nombre de relación por defecto para obtener los cultivos asociados
    crops = field.crops.all()
    has_crops = Crop.objects.filter(field__scout=scout_id).exists()

    return render(request, 'field_detail.html', {'field': field, 'crops': crops, 'has_crops': has_crops})


def crop_detail(request, crop_id):
    crop = get_object_or_404(Crop, id=crop_id)
    soil_test = crop.field.soil_tests.latest('date')

    # Generar recomendaciones basadas en el análisis de suelo y el cultivo
    recommendations = Recommendation.objects.filter(crop=crop)

    if request.method == 'POST':
        form = CropObservationForm(request.POST)
        if form.is_valid():
            observation = form.save(commit=False)
            observation.crop = crop
            observation.save()
            return redirect('crop-detail', crop_id=crop.id)
    else:
        form = CropObservationForm()

    # Renderizar la plantilla con todas las recomendaciones
    return render(request, 'crop_detail.html', {
        'crop': crop,
        'form': form,
        # Pasar todas las recomendaciones a la plantilla
        'recommendations': recommendations,
    })


def create_observation(request, crop_id):
    crop = get_object_or_404(Crop, id=crop_id)
    if request.method == 'POST':
        form = CropObservationForm(request.POST)
        if form.is_valid():
            observation = form.save(commit=False)
            observation.crop = crop
            observation.save()

            # Verificar que los datos de `soil_test` existan
            # Ajustar la lógica si es necesario
            soil_test = SoilTest.objects.filter(crop=crop).first()

            if soil_test:
                try:
                    generate_recommendations(soil_test, crop, observation)
                except Exception as e:
                    print(f"Error al generar la recomendación: {e}")

            return redirect('crop-detail', crop_id=crop.id)
    else:
        form = CropObservationForm()
    return render(request, 'crop_detail.html', {'crop': crop, 'form': form})


def edit_observation(request, observation_id):
    observation = get_object_or_404(CropObservation, id=observation_id)
    if request.method == 'POST':
        form = CropObservationForm(request.POST, instance=observation)
        if form.is_valid():
            form.save()
            return redirect('crop-detail', crop_id=observation.crop.id)
    else:
        form = CropObservationForm(instance=observation)
    return render(request, 'edit_observation.html', {'form': form, 'observation': observation})


def delete_observation(request, observation_id):
    observation = get_object_or_404(CropObservation, id=observation_id)
    crop_id = observation.crop.id
    if request.method == 'POST':
        observation.delete()
        return redirect('crop-detail', crop_id=crop_id)
    return render(request, 'confirm_delete_observation.html', {'observation': observation})


def crop_growth_data(request, crop_id):
    crop = get_object_or_404(Crop, pk=crop_id)
    observations = CropObservation.objects.filter(crop=crop).order_by('date')

    data = {
        "dates": [obs.date.strftime("%Y-%m-%d") for obs in observations],
        "heights": [obs.crop_height for obs in observations],
        "soil_temperatures": [obs.soil_temperature for obs in observations],
        "weed_pressures": [obs.weed_pressure for obs in observations],
        "weed_heights": [obs.weed_height for obs in observations],
        "rain_accumulations": [obs.rain_accumulation for obs in observations]
    }

    return JsonResponse(data)


def get_weather(request, field_id):
    field = get_object_or_404(Field, id=field_id)
    # Reemplaza con tu clave de API de OpenWeatherMap
    api_key = '2c64e7bce7cc4a01042fe3f2d66604e5'

    # Obtener el idioma del usuario desde la solicitud
    # Usa 'en' como predeterminado si no se encuentra el idioma
    user_language = request.LANGUAGE_CODE or 'en'

    url = f"http://api.openweathermap.org/data/2.5/weather?lat={field.latitude}&lon={field.longitude}&appid={api_key}&units=metric&lang={user_language}"

    response = requests.get(url)
    weather_data = response.json()

    if response.status_code == 200:
        data = {
            'temperature': weather_data['main']['temp'],
            'description': weather_data['weather'][0]['description'],
            'location': field.location
        }
    else:
        data = {'error': 'No se pudo obtener el clima para esta ubicación.'}

    return JsonResponse(data)


@user_required('scout')
def field_area_comparison(request):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    # Filtrar campos por scout autenticado
    fields = Field.objects.filter(scout=scout_id)
    names = [field.name for field in fields]
    # Suponiendo que size es el campo que representa el tamaño en hectáreas
    areas = [field.size for field in fields]

    data = {
        'names': names,
        'areas': areas,
    }

    return JsonResponse(data)


@user_required('scout')
def field_climate_analysis(request):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    # Filtrar campos por scout autenticado
    fields = Field.objects.filter(scout=scout_id)

    data = []
    for field in fields:
        field_data = {
            'name': field.name,
            # Simulación de temperatura entre 15 y 25 grados Celsius
            'temperature': round(15 + 10 * random.random(), 1),
            # Simulación de humedad entre 50% y 80%
            'humidity': round(50 + 30 * random.random(), 1),
            # Simulación de precipitación entre 0 y 10 mm
            'precipitation': round(0 + 10 * random.random(), 1)
        }
        data.append(field_data)

    return JsonResponse(data, safe=False)


@user_required('scout')
def harvest_date_distribution(request):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    crops = Crop.objects.filter(field__scout=scout_id)
    harvest_dates = [crop.expected_harvest_date.strftime(
        '%Y-%m-%d') for crop in crops if crop.expected_harvest_date]

    date_counts = Counter(harvest_dates)
    sorted_dates = sorted(date_counts.items())

    data = {
        'dates': [date for date, count in sorted_dates],
        'counts': [count for date, count in sorted_dates]
    }

    return JsonResponse(data)


@user_required('scout')
def soil_type_distribution(request):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    # Obtener la distribución de tipos de suelo
    soil_data = Field.objects.filter(scout=scout_id).values(
        'soil_type').annotate(total_area=Sum('size'))

    # Formatear los datos para el gráfico
    soil_types = [soil['soil_type'] for soil in soil_data]
    soil_areas = [soil['total_area'] for soil in soil_data]

    return JsonResponse({'soil_types': soil_types, 'soil_areas': soil_areas})


def crop_distribution_data(request, field_id):
    crops = Crop.objects.filter(field_id=field_id)
    crop_types = crops.values_list('crop_type', flat=True).distinct()

    data = []
    for crop_type in crop_types:
        count = crops.filter(crop_type=crop_type).count()
        data.append({'crop_type': crop_type, 'count': count})

    return JsonResponse(data, safe=False)


def field_size_comparison_view(request):
    fields = Field.objects.all()
    data = [{'field_name': field.name, 'size': field.size} for field in fields]
    return JsonResponse(data, safe=False)


def get_crop_types_data(request, field_id):
    field = get_object_or_404(Field, id=field_id)
    crops = field.crops.all()
    crop_types = crops.values('crop_type').annotate(count=Count('crop_type'))
    data = {
        'labels': [dict(Crop.CROP_TYPE_CHOICES).get(ctype['crop_type'], 'Desconocido') for ctype in crop_types],
        'counts': [ctype['count'] for ctype in crop_types]
    }
    return JsonResponse(data)


def get_growth_trend_data(request, field_id):
    field = get_object_or_404(Field, id=field_id)
    crops = field.crops.all()
    observations = CropObservation.objects.filter(
        crop__in=crops).order_by('date')
    dates = observations.values_list('date', flat=True).distinct()
    average_heights = [observations.filter(date=date).aggregate(
        Avg('crop_height'))['crop_height__avg'] for date in dates]
    data = {
        'dates': list(dates),
        'average_heights': average_heights
    }
    return JsonResponse(data)


def get_soil_climate_data(request, field_id):
    field = get_object_or_404(Field, id=field_id)
    crops = field.crops.all()
    observations = CropObservation.objects.filter(crop__in=crops)

    # Asegúrate de que realmente hay observaciones
    if not observations.exists():
        return JsonResponse({
            'soil_conditions': [],
            'heights': []
        })

    soil_conditions = observations.values(
        'date', 'soil_temperature', 'crop_height')
    data = {
        'dates': [obs['date'] for obs in soil_conditions],
        'soil_temperatures': [obs['soil_temperature'] for obs in soil_conditions],
        'crop_heights': [obs['crop_height'] for obs in soil_conditions]
    }
    return JsonResponse(data)


# def add_crop(request):
#     if request.method == 'POST':
#         form = CropForm(request.POST)

#         if form.is_valid():
#             field_id = form.cleaned_data.get('field_id')

#             if not field_id:
#                 messages.error(request, _('ID del campo no proporcionado.'))
#                 return redirect('field-list')  # Redirige a una página de lista de campos si no se proporciona el ID

#             try:
#                 field = Field.objects.get(id=field_id)
#             except Field.DoesNotExist:
#                 messages.error(request, 'Campo no encontrado.')
#                 return redirect('field-list')  # Redirige a una página de lista de campos si el campo no existe

#             crop = form.save(commit=False)
#             crop.field = field
#             crop.save()

#             messages.success(request, _('Cultivo agregado exitosamente.'))
#             return redirect('field-detail', field_id=field_id)  # Redirige a la página de detalles del campo

#         else:
#             messages.error(request, _('Error al agregar el cultivo.'))
#             field_id = request.POST.get('field_id')
#             if not field_id:
#                 return redirect('field-list')  # Redirige a una página de lista de campos si no se proporciona el ID
#             return redirect('field-detail', field_id=field_id)  # Redirige a la página de detalles del campo con el formulario incompleto

#     else:
#         # Maneja la solicitud GET
#         field_id = request.GET.get('field_id')
#         if not field_id:
#             messages.error(request, _('ID del campo no proporcionado.'))
#             return redirect('field-list')  # Redirige a una página de lista de campos si no se proporciona el ID

#         # Redirige a la vista de detalles del campo
#         return redirect('field-detail', field_id=field_id)

@user_required('scout')
def dashboard(request):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    # Gráficos de cultivos
    crops = Crop.objects.filter(field__scout=scout_id)

    # Agrupación mensual de alturas de cultivos
    monthly_data = crops.annotate(
        month=TruncMonth('observations__date')  # Agrupar por mes
    ).values('month').annotate(
        avg_height=Avg('observations__crop_height')  # Promediar la altura
    ).order_by('month')

    harvest_chart_data = {
        'dates': [data['month'].strftime('%Y-%m') for data in monthly_data if data['month'] is not None],
        'avg_heights': [data['avg_height'] for data in monthly_data if data['month'] is not None]
    }

    # Próximas cosechas
    today = datetime.today().date()
    upcoming_harvests = Crop.objects.filter(
        field__scout=scout_id,
        expected_harvest_date__gte=today,
        expected_harvest_date__lte=today + timedelta(days=30)
    ).order_by('expected_harvest_date')

    # Información del foro social
    recent_posts = Post.objects.all().order_by(
        '-created_at')[:5]  # Últimas 5 publicaciones

    # Recomendaciones (ajustada)
    recommendations = Recommendation.objects.filter(
        crop_observation__crop__field__scout=scout_id)

    # Noticias
    news_items = News.objects.all().order_by(
        '-created_at')[:5]  # Últimas 5 noticias

    # Recupera todos los campos asociados al Scout
    fields = Field.objects.filter(scout_id=scout_id)
    if not fields.exists():
        return render(request, 'dashboard.html', {'message': _('No hay campos asociados al scout.')})

    crops = Crop.objects.filter(field__in=fields)
    if not crops.exists():
        return render(request, 'dashboard.html', {'message': _('No hay cultivos en los campos del scout.')})

    projected_gains = []

    for crop in crops:
        num_plants = crop.num_plants or 0
        yield_per_plant = crop.yield_per_plant or 0
        market_price = crop.market_price or 0

        estimated_production = num_plants * yield_per_plant
        estimated_income = estimated_production * market_price

        projected_gains.append({
            'crop_name': crop.name,
            'num_plants': num_plants,
            'estimated_production': estimated_production,
            'market_price': market_price,
            'estimated_income': estimated_income,
        })

    # Comprobar si hay cultivos asociados a los campos del scout
    has_crops = Crop.objects.filter(field__scout=scout_id).exists()

    context = {
        'harvest_chart_data': harvest_chart_data,
        'upcoming_harvests': upcoming_harvests,
        'recent_posts': recent_posts,
        'recommendations': recommendations,
        'news_items': news_items,
        'projected_gains': projected_gains,
        'scout': scout_id,
        'has_crops': has_crops,
    }

    return render(request, 'dashboard.html', context)


@user_required('scout')
def crop_growth_chart(request):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    # Crear un diccionario para almacenar el tamaño acumulado de las plantas por cultivo y por mes
    growth_data = defaultdict(lambda: defaultdict(float))

    # Obtener todas las observaciones de los cultivos del Scout actual
    observations = CropObservation.objects.filter(crop__field__scout=scout_id)

    # Organizar los datos por cultivo y por mes
    for observation in observations:
        crop_name = observation.crop.name
        month = observation.date.strftime('%Y-%m')
        growth_data[crop_name][month] += observation.crop_height

    # Preparar los datos para el gráfico
    series = []
    categories = sorted(set(month for crop_data in growth_data.values()
                        for month in crop_data.keys()))

    for crop_name, crop_data in growth_data.items():
        series.append({
            'name': crop_name,
            'data': [crop_data.get(month, 0) for month in categories]
        })

    # Verifica el contenido de la respuesta antes de enviarla
    if not series or not categories:
        return JsonResponse({'error': _('No hay datos disponibles')}, status=404)

    return JsonResponse({'categories': categories, 'series': series})


def dashboard_view(request):
    return render(request, 'dashboard.html')


@user_required('scout')
def recommendations_view(request):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    # Filtrar recomendaciones basadas en las observaciones de cultivos del scout actual
    recommendations = Recommendation.objects.filter(
        crop_observation__crop__field__scout=scout_id)

    context = {
        'recommendations': recommendations
    }

    return render(request, 'recommendations.html', context)


def notifications_view(request):
    # Notificaciones específicas para el usuario actual (recomendaciones)
    user_notifications = Notification.objects.filter(
        user=request.user, type='fertilization', status='unread')

    # Notificaciones generales de noticias (visibles para todos los usuarios)
    general_notifications = Notification.objects.filter(
        type='news', status='unread')

    # Combinar ambas listas de notificaciones usando '|'
    notifications = user_notifications | general_notifications

    return render(request, 'layout.html', {'notifications': notifications})


def mark_all_read(request):
    Notification.objects.filter(
        user=request.user, status='unread').update(status='read')
    return redirect('dashboard')


@user_required('scout')
def edit_scout_profile(request, scout_id):
    scout = get_current_scout(request)
    scout = Scout.objects.get(id=scout_id)

    if request.method == 'POST':
        form = ScoutProfileForm(request.POST, request.FILES, instance=scout)
        if form.is_valid():
            form.save()
            messages.success(request, 'Perfil actualizado con éxito.')
            return redirect('edit_scout_profile', scout_id=scout.id)
    else:
        form = ScoutProfileForm(instance=scout)

    return render(request, 'edit_scout_profile.html', {'form': form})


def get_plant_count_history(request, crop_id):
    crop = get_object_or_404(Crop, id=crop_id)

    # Obtener todas las observaciones del cultivo, ordenadas por fecha
    observations = CropObservation.objects.filter(crop=crop).order_by('date')

    # Preparar los datos para el gráfico
    data = {
        # Fechas de observación
        "dates": [obs.date.strftime("%Y-%m-%d") for obs in observations],
        # Cantidad de plantas en cada observación
        "plant_counts": [obs.stand_count for obs in observations]
    }

    return JsonResponse(data)


@user_required('scout')
def create_crop(rsequest):
    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    if request.method == 'POST':
        form = CropForm(request.POST)
        if form.is_valid():
            # Crear la instancia pero no guardar aún
            crop = form.save(commit=False)

            # Capturar los valores de nutrientes
            nitrogen = request.POST.get('nitrogen')
            phosphorus = request.POST.get('phosphorus')
            potassium = request.POST.get('potassium')

            # Convertirlos a JSON
            crop.nutrient_requirements = {
                "N": int(nitrogen),
                "P": int(phosphorus),
                "K": int(potassium)
            }

            crop.scout = scout_id  # Asignar el scout actual si es necesario
            crop.save()  # Guardar el cultivo
            messages.success(request, "Cultivo creado con éxito.")
            return redirect('crop-list')
    else:
        form = CropForm()

    # Proveer campos al formulario si se necesita
    fields = Field.objects.filter(scout_id=scout_id)
    return render(request, 'create_crop.html', {'form': form, 'fields': fields})


@csrf_exempt
def count_plants(request):
    if request.method == 'POST':
        # Código para manejar la imagen
        data = json.loads(request.body)
        image_data = data['image'].split(',')[1]
        try:
            image = Image.open(io.BytesIO(base64.b64decode(image_data)))
            open_cv_image = cv2.cvtColor(np.array(image), cv2.COLOR_RGB2BGR)
            plant_count = count_plants_in_image(open_cv_image)
            return JsonResponse({'num_plants': plant_count})
        except Exception as e:
            return JsonResponse({'error': str(e)}, status=400)
    return JsonResponse({'error': 'Método no permitido'}, status=405)


def count_plants_in_image(image):
    # Convertir la imagen al espacio de color HSV para facilitar el filtrado de color
    hsv = cv2.cvtColor(image, cv2.COLOR_BGR2HSV)

    # Definir los rangos de color para detectar el verde de las plantas
    # Ajusta según las condiciones de la imagen
    lower_green = np.array([25, 40, 40])
    upper_green = np.array([90, 255, 255])

    # Crear una máscara que aísle los colores verdes
    mask = cv2.inRange(hsv, lower_green, upper_green)

    # Aplicar un desenfoque para reducir el ruido
    blurred = cv2.GaussianBlur(mask, (5, 5), 0)

    # Operaciones morfológicas para limpiar la imagen binaria
    kernel = np.ones((3, 3), np.uint8)
    closing = cv2.morphologyEx(blurred, cv2.MORPH_CLOSE, kernel, iterations=2)
    opening = cv2.morphologyEx(closing, cv2.MORPH_OPEN, kernel, iterations=1)

    # Encontrar contornos en la imagen procesada
    contours, _ = cv2.findContours(
        opening, cv2.RETR_EXTERNAL, cv2.CHAIN_APPROX_SIMPLE)

    # Filtrar contornos por área mínima para evitar contar ruido (ajustar según la imagen)
    filtered_contours = [cnt for cnt in contours if cv2.contourArea(cnt) > 100]

    return len(filtered_contours)  # Asegúrate de que esto devuelva un número


def get_subscriber_hash(email):
    """Calcular el hash MD5 del email para Mailchimp."""
    return hashlib.md5(email.lower().encode('utf-8')).hexdigest()


def subscribe_mailchimp(request):
    if request.method == 'POST':
        email = request.POST.get('email')
        if not email:
            messages.error(
                request, "Por favor, ingresa un correo electrónico válido.")
            return redirect('landing-page')  # Página de regreso

        try:
            # Configurar Mailchimp Client
            client = Client()
            client.set_config({
                "api_key": settings.MAILCHIMP_API_KEY,
                "server": settings.MAILCHIMP_SERVER_PREFIX
            })

            # Calcular el hash del suscriptor
            subscriber_hash = get_subscriber_hash(email)

            # Añadir o actualizar miembro en la lista
            client.lists.set_list_member(
                settings.MAILCHIMP_LIST_ID,
                subscriber_hash,
                {
                    "email_address": email,
                    "status_if_new": "subscribed",
                }
            )

            # Agregar etiqueta "waiting-list"
            client.lists.update_list_member_tags(
                settings.MAILCHIMP_LIST_ID,
                subscriber_hash,
                body={
                    "tags": [{"name": "waiting-list", "status": "active"}]
                }
            )

            # Mostrar mensaje de éxito
            messages.success(
                request, "¡Te has registrado exitosamente en la lista de espera!")
        except ApiClientError as error:
            logger.error(f"Error en Mailchimp: {error.text}")
            if 'Member Exists' in error.text:
                messages.info(
                    request, "Este correo ya está registrado en la lista de espera.")
            else:
                messages.error(
                    request, "Hubo un error al procesar tu solicitud. Inténtalo de nuevo más tarde.")

        # Redirigir a la página principal
        return redirect('landing-page')

    return redirect('landing-page')  # Redirigir si no es POST
