import base64
import io
import json
import logging
import random
import string
from collections import Counter, defaultdict
from datetime import datetime, timedelta
import hashlib

# Librerías de terceros para manejo de imágenes y solicitudes HTTP
import cv2
import numpy as np
import requests
from PIL import Image
from ultralytics import YOLO
import cv2
import numpy as np
import json
import base64
from django.http import JsonResponse
from django.views.decorators.csrf import csrf_exempt
import torch
from inference_sdk import InferenceHTTPClient
import tempfile

# Librerias de Django
from django.conf import settings
from django.contrib import messages
from django.contrib.auth import get_user_model, logout
from django.contrib.auth.decorators import login_required
from django.contrib.contenttypes.models import ContentType
from django.core.exceptions import ObjectDoesNotExist
from django.core.mail import send_mail
from django.db.models import Avg, Count, Sum
from django.db.models.functions import TruncMonth
from django.http import JsonResponse
from django.shortcuts import get_object_or_404, redirect, render
from django.template.loader import render_to_string
from django.utils import timezone
from django.utils.formats import sanitize_separators
from django.utils.html import strip_tags
from django.utils.translation import gettext as _
from django.views import View
from django.views.decorators.csrf import csrf_exempt


# Mailchimp
import mailchimp_marketing as MailchimpMarketing
from mailchimp_marketing import Client
from mailchimp_marketing.api_client import ApiClientError

# Decoradores locales
from agroapp.common.decorators import user_required

# Formularios
from .forms import (
    CropForm, CropObservationForm, FieldForm, ScoutProfileForm, SoilTestForm
)


from .forms import UserRegistrationForm, ScoutRegistrationForm, FarmerRegistrationForm


# Modelos
from .models import (
    Crop, CropObservation, Field, Recommendation, Scout, SoilTest, Token, Notification, SoilType
)

# Utils
from .utils import generate_recommendations, get_current_scout, generate_soil_recommendations

# Modelos de otras aplicaciones
from social.models import News
from social_network.models import Comment, Post, Book, Video
from social_network.forms import PostForm, CommentForm
from farmer.models import Farmer
from agronomist.models import Agronomist


# Estas importaciones ya están cubiertas en las secciones anteriores, pero se repiten aquí para claridad
from django.shortcuts import render, redirect
from django.contrib.contenttypes.models import ContentType
from django.utils.translation import gettext as _
from django.utils.timezone import now
import logging

# Logger
User = get_user_model()
logger = logging.getLogger(__name__)


def scout_list(request):
    scouts = Scout.objects.all()
    return render(request, 'scouts/scout_list.html', {'scouts': scouts})

# Función para generar un token de 6 dígitos


def generate_token():
    return ''.join(random.choices(string.digits, k=6))

# Función para enviar el correo con el token


# def send_verification_email(user, token_value):
#     html_message = render_to_string(
#         'email/verification_token.html', {'token': token_value})
#     plain_message = strip_tags(html_message)
#     subject = _('Tu código de verificación')
#     from_email = 'AgriCarm App<no-reply@agricarm.com>'
#     recipient_list = [user.email]

#     send_mail(subject, plain_message, from_email,
#               recipient_list, html_message=html_message)

import requests

def send_verification_email(user, token_value):
    webhook_url = 'https://hook.us2.make.com/l121va7chxagx4wy7bx0sgxexdtrls8y'  # Reemplaza con tu URL real

    payload = {
        'email': user.email,
        'nombre': user.first_name,  # o cualquier otro dato útil
        'codigo': token_value
    }

    try:
        response = requests.post(webhook_url, json=payload)
        response.raise_for_status()
    except requests.RequestException as e:
        # Loguea o maneja el error según tu necesidad
        print(f'Error enviando al webhook: {e}')



# Función de registro mejorada
def register(request):
    if request.method == 'POST':
        user_type_form = UserRegistrationForm(request.POST)

        if user_type_form.is_valid():
            user_type = user_type_form.cleaned_data['user_type']

            if user_type == 'scout':
                form = ScoutRegistrationForm(request.POST)
                if form.is_valid():
                    scout = form.save(commit=False)
                    scout.user_type = 'scout'
                    scout.save()

                    # Generar y guardar el token de verificación
                    token_value = generate_token()
                    token = Token.objects.create(
                        content_type=ContentType.objects.get_for_model(Scout),
                        object_id=scout.id,
                        token=token_value
                    )

                    # Enviar el correo con el token
                    send_verification_email(scout, token_value)

                    request.session['email'] = scout.email
                    return redirect('verify')

            elif user_type == 'farmer':
                form = FarmerRegistrationForm(request.POST)
                if form.is_valid():
                    farmer = form.save(commit=False)
                    farmer.save()

                    # Generar y guardar el token de verificación para Farmers
                    token_value = generate_token()
                    token = Token.objects.create(
                        content_type=ContentType.objects.get_for_model(Farmer),
                        object_id=farmer.id,
                        token=token_value
                    )

                    # Enviar el correo con el token
                    send_verification_email(farmer, token_value)

                    request.session['email'] = farmer.email
                    return redirect('verify')

    else:
        user_type_form = UserRegistrationForm()
        form = None

    return render(request, 'registro.html', {
        'user_type_form': user_type_form,
        'form': form
    })


# Función de login adaptada para múltiples tipos de usuarios
def login_request(request):
    if request.method == 'POST':
        email = request.POST['email']
        try:
            # Buscar al usuario en los tres modelos posibles
            user = None
            user_model = None

            if Scout.objects.filter(email=email).exists():
                user = Scout.objects.get(email=email)
                user_model = Scout

            elif Farmer.objects.filter(email=email).exists():
                user = Farmer.objects.get(email=email)
                user_model = Farmer

            elif Agronomist.objects.filter(email=email).exists():
                user = Agronomist.objects.get(email=email)
                user_model = Agronomist

            if not user:
                return render(request, 'login.html', {'error': _('Correo electrónico no encontrado')})

            # Generar el token
            token_value = generate_token()

            # Crear un token vinculado genéricamente al usuario
            content_type = ContentType.objects.get_for_model(user_model)
            Token.objects.create(content_type=content_type,
                                 object_id=user.id, token=token_value)

            # Enviar el correo con el token
            send_verification_email(user, token_value)

            # Guardar el email en la sesión para usarlo en la verificación
            request.session['email'] = email
            # Redirige a la página de verificación de código
            return redirect('verify')

        except Exception as e:
            return render(request, 'login.html', {'error': _('Ocurrió un error: ') + str(e)})

    return render(request, 'login.html')


def verify_token(request):
    if request.method == 'POST':
        email = request.session.get('email')

        # Recolecta los dígitos individuales del formulario
        token_value = ''.join([
            request.POST.get(f'digit{i}', '') for i in range(1, 7)
        ])

        logger.info(
            f'🔍 Iniciando verificación de token para: {email}, token ingresado: {token_value}')

        try:
            # Lista de modelos y dashboards correspondientes
            user_types = [
                (Scout, 'dashboard'),
                (Farmer, 'farmer_dashboard'),
                (Agronomist, 'agronomist-dashboard'),
            ]

            for model, dashboard in user_types:
                logger.info(f'🛠️ Verificando modelo: {model.__name__}')

                # Buscar el usuario en el modelo actual
                user = model.objects.filter(email=email).first()
                if not user:
                    logger.warning(
                        f'⚠️ Usuario NO encontrado en {model.__name__} para email: {email}')
                    continue

                logger.info(
                    f'✅ Usuario encontrado en {model.__name__}: {user.id}')

                # Buscar el token asociado al usuario
                try:
                    content_type = ContentType.objects.get_for_model(model)
                    token = Token.objects.filter(
                        token=token_value,
                        content_type=content_type,
                        object_id=user.id
                    ).order_by('-created_at').first()
                except Exception as e:
                    logger.error(
                        f'🔥 ERROR al buscar token en {model.__name__}: {e}')
                    continue

                if token and token.is_valid():
                    request.session[f'{model.__name__.lower()}_id'] = user.id
                    logger.info(
                        f'✅ Token verificado con éxito para {model.__name__}: {user.id}')
                    return redirect(dashboard)
                else:
                    logger.warning(
                        f'⚠️ Token inválido o expirado para {model.__name__} y usuario ID {user.id}')

            # Si no se encuentra un token válido
            logger.error(
                f'❌ Ningún token válido encontrado para email: {email}')
            return render(request, 'verificar.html', {'error': _('Token inválido o expirado')})

        except Exception as e:
            logger.error(f'🔥 Error inesperado en verify_token: {e}')
            return render(request, 'verificar.html', {'error': _('Ocurrió un error inesperado')})

    logger.info('📥 GET request recibido en verify_token')
    return render(request, 'verificar.html')


def logout_view(request):
    logout(request)  # Esta función elimina toda la información de la sesión
    # Redirige a la página de login después del logout
    return redirect('login')


@user_required('scout')
def fieldlistcreate(request):
    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    # Obtener la instancia del Scout usando el ID almacenado en la sesión
    try:
        scout = Scout.objects.get(id=scout_id)
    except Scout.DoesNotExist:
        raise ValueError("El Scout con este ID no existe en la base de datos")

    if request.method == 'POST':
        form = FieldForm(request.POST)
        if form.is_valid():
            field = form.save(commit=False)

            # ✅ Asignar el Scout al campo
            field.scout = scout

            # ✅ Asignar el Farmer desde la relación del Scout
            if scout.farmer:
                field.farmer = scout.farmer
            else:
                messages.error(
                    request, "El Scout no está asignado a ningún Finquero.")
                return redirect('field-list')

            field.save()
            return redirect('field-list')  # Redirige a la lista de Fields

    else:
        form = FieldForm()

    # Filtramos usando la instancia real
    fields = Field.objects.filter(scout=scout)
    soil_types = SoilType.objects.all()  # ✅ Obtener los tipos de suelo disponibles

    # Preparar datos para el gráfico
    fields_data = [{
        'name': field.name,
        'area': field.size
    } for field in fields]

    has_crops = Crop.objects.filter(field__scout=scout).exists()

    return render(request, 'field_list.html', {
        'fields': fields,
        'form': form,
        'soil_types': soil_types,  # ✅ Pasamos los tipos de suelo al template
        'fields_data': fields_data,
        'has_crops': has_crops
    })


@user_required('scout')
def edit_field(request, field_id):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    field = get_object_or_404(Field, id=field_id, scout_id=scout_id)
    # Depuración
    print(
        f"Datos del campo: {field.latitude}, {field.longitude}, {field.size}")

    soil_types = SoilType.objects.all()  # ✅ Obtener todos los tipos de suelo

    if request.method == 'POST':
        form = FieldForm(request.POST, instance=field)
        if form.is_valid():
            form.save()
            return redirect('field-list')  # Redirige a la lista de campos
    else:
        form = FieldForm(instance=field)

    return render(request, 'field_list.html', {'form': form, 'field': field})


@user_required('scout')
def delete_field(request, field_id):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    field = get_object_or_404(Field, id=field_id, scout=scout_id)

    if request.method == 'GET':
        field.delete()
        # Redirige a la lista de campos después de la eliminación
        return redirect('field-list')

    return render(request, 'field_list.html', {'field': field})


@user_required('scout')
def crop_list(request):
    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    # Filtrar cultivos y campos relacionados al Scout actual
    fields = Field.objects.filter(scout_id=scout_id)
    crops = Crop.objects.filter(field__scout=scout_id)
    has_crops = crops.exists()
    has_fields = fields.exists()

    return render(request, 'crop_list.html', {
        'crops': crops,
        'scout': scout_id,
        'has_crops': has_crops,
        'has_fields': has_fields,
        'fields': fields,
    })


@user_required('scout')
def crop_type_distribution(request):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    # Filtrar los cultivos del scout autenticado
    crops = Crop.objects.filter(field__scout=scout_id)

    # Verificar si el scout tiene cultivos asociados
    if not crops.exists():
        return JsonResponse({
            'error': _('No hay cultivos asociados a este scout.')
        }, status=404)

    # Obtener el conteo de cultivos por tipo
    crop_types_count = crops.values(
        'crop_type').annotate(count=Count('crop_type'))

    # Verificar si se encontraron tipos de cultivos
    if not crop_types_count:
        return JsonResponse({
            'error': _('No hay datos de tipos de cultivos.')
        }, status=404)

    # Mapear los nombres amigables de los tipos de cultivos
    crop_types_labels = [dict(Crop.CROP_TYPE_CHOICES).get(
        ctype['crop_type'], _('Desconocido')) for ctype in crop_types_count]
    crop_types_data = [ctype['count'] for ctype in crop_types_count]

    # Verificar si se recuperaron datos
    if not crop_types_labels or not crop_types_data:
        return JsonResponse({
            'error': _('No se encontraron datos para los tipos de cultivos.')
        }, status=404)

    # Devolver la información en formato JSON
    return JsonResponse({
        'labels': crop_types_labels,
        'data': crop_types_data,
    })


@user_required('scout')
def edit_crop(request, crop_id):
    crop = get_object_or_404(Crop, id=crop_id)

    # Obtener opciones de tipo de cultivo
    crop_type_choices = Crop.CROP_TYPE_CHOICES

    # Obtener todos los campos relacionados con el Scout
    fields = Field.objects.filter(scout=request.session.get('scout_id'))

    if request.method == 'POST':
        form = CropForm(request.POST, instance=crop)
        if form.is_valid():
            # Procesar datos de nutrientes
            # nitrogen = request.POST.get('nitrogen', 0)
            # phosphorus = request.POST.get('phosphorus', 0)
            # potassium = request.POST.get('potassium', 0)

            crop = form.save(commit=False)

            # Procesar datos de nutrientes y evitar localización
            nitrogen = sanitize_separators(request.POST.get('nitrogen', '0'))
            phosphorus = sanitize_separators(
                request.POST.get('phosphorus', '0'))
            potassium = sanitize_separators(request.POST.get('potassium', '0'))

            crop.nutrient_requirements = {
                "N": float(nitrogen),
                "P": float(phosphorus),
                "K": float(potassium),
            }
            crop.save()
            return redirect('crop-list')
    else:
        form = CropForm(instance=crop)

    return render(request, 'edit_crop.html', {
        'form': form,
        'crop': crop,
        'crop_type_choices': crop_type_choices,
        'fields': fields,
    })


@user_required('scout')
def delete_crop(request, crop_id):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    crop = get_object_or_404(Crop, id=crop_id, field__scout=scout_id)

    if request.method == 'GET':
        crop.delete()  # Eliminar el cultivo
        # Redirige a la lista de cultivos después de eliminar
        return redirect('crop-list')

    # Para cualquier otra solicitud, redirigir a la lista de cultivos
    return redirect('crop-list')


@user_required('scout')
def field_detail(request, field_id):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    field = get_object_or_404(Field, id=field_id)
    # Usar el nombre de relación por defecto para obtener los cultivos asociados
    crops = field.crops.all()
    has_crops = Crop.objects.filter(field__scout=scout_id).exists()

    return render(request, 'field_detail.html', {'field': field, 'crops': crops, 'has_crops': has_crops})


def crop_detail(request, crop_id):
    crop = get_object_or_404(Crop, id=crop_id)
    soil_test = crop.field.soil_tests.latest(
        'date') if hasattr(crop.field, 'soil_tests') else None
    recommendations = Recommendation.objects.filter(crop=crop)

    # Obtener el número de plantas
    if crop.observations.exists():
        last_observation = crop.observations.latest('date')
        num_plants = last_observation.stand_count
    else:
        num_plants = crop.num_plants

    if request.method == 'POST':
        form = CropObservationForm(request.POST)
        if form.is_valid():
            observation = form.save(commit=False)
            observation.crop = crop
            observation.save()
            return redirect('crop-detail', crop_id=crop.id)
    else:
        form = CropObservationForm(
            initial={'num_plants': num_plants})  # Inicializar el campo

    return render(request, 'crop_detail.html', {
        'crop': crop,
        'form': form,
        'recommendations': recommendations,
        'num_plants': num_plants,
        'soil_test': soil_test,
    })


@user_required('scout')
def create_observation(request, crop_id):

    scout = request.session.get('scout_id')
    if not scout:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    crop = get_object_or_404(Crop, id=crop_id)

    if request.method == 'POST':
        form = CropObservationForm(request.POST)
        if form.is_valid():
            observation = form.save(commit=False)
            observation.crop = crop
            observation.save()

            # Obtener el análisis de suelo relacionado con el campo del cultivo
            soil_test = SoilTest.objects.filter(field=crop.field).first()

            if soil_test:
                try:
                    generate_recommendations(soil_test, crop, observation)
                except Exception as e:
                    print(f"Error al generar la recomendación: {e}")

            return redirect('crop-detail', crop_id=crop.id)
    else:
        form = CropObservationForm()

    return render(request, 'crop_detail.html', {'crop': crop, 'form': form})


def edit_observation(request, observation_id):
    observation = get_object_or_404(CropObservation, id=observation_id)
    if request.method == 'POST':
        form = CropObservationForm(request.POST, instance=observation)
        if form.is_valid():
            form.save()
            return redirect('crop-detail', crop_id=observation.crop.id)
    else:
        form = CropObservationForm(instance=observation)
    return render(request, 'edit_observation.html', {'form': form, 'observation': observation})


def delete_observation(request, observation_id):
    observation = get_object_or_404(CropObservation, id=observation_id)
    crop_id = observation.crop.id
    if request.method == 'POST':
        observation.delete()
        return redirect('crop-detail', crop_id=crop_id)
    return render(request, 'confirm_delete_observation.html', {'observation': observation})


def crop_growth_data(request, crop_id):
    crop = get_object_or_404(Crop, pk=crop_id)
    observations = CropObservation.objects.filter(crop=crop).order_by('date')

    data = {
        "dates": [obs.date.strftime("%Y-%m-%d") for obs in observations],
        "heights": [obs.crop_height for obs in observations],
        "soil_temperatures": [obs.soil_temperature for obs in observations],
        "weed_pressures": [obs.weed_pressure for obs in observations],
        "weed_heights": [obs.weed_height for obs in observations],
        "rain_accumulations": [obs.rain_accumulation for obs in observations]
    }

    return JsonResponse(data)


def get_weather(request, field_id):
    field = get_object_or_404(Field, id=field_id)
    # Reemplaza con tu clave de API de OpenWeatherMap
    api_key = '2c64e7bce7cc4a01042fe3f2d66604e5'

    # Obtener el idioma del usuario desde la solicitud
    # Usa 'en' como predeterminado si no se encuentra el idioma
    user_language = request.LANGUAGE_CODE or 'en'

    url = f"http://api.openweathermap.org/data/2.5/weather?lat={field.latitude}&lon={field.longitude}&appid={api_key}&units=metric&lang={user_language}"

    response = requests.get(url)
    weather_data = response.json()

    if response.status_code == 200:
        data = {
            'temperature': weather_data['main']['temp'],
            'description': weather_data['weather'][0]['description'],
            'location': field.location
        }
    else:
        data = {'error': _('No se pudo obtener el clima para esta ubicación.')}

    return JsonResponse(data)


@user_required('scout')
def field_area_comparison(request):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    # Filtrar campos por scout autenticado
    fields = Field.objects.filter(scout=scout_id)
    names = [field.name for field in fields]
    # Suponiendo que size es el campo que representa el tamaño en hectáreas
    areas = [field.size for field in fields]

    data = {
        'names': names,
        'areas': areas,
    }

    return JsonResponse(data)


@user_required('scout')
def field_climate_analysis(request):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    # Filtrar campos por scout autenticado
    fields = Field.objects.filter(scout=scout_id)

    data = []
    for field in fields:
        field_data = {
            'name': field.name,
            # Simulación de temperatura entre 15 y 25 grados Celsius
            'temperature': round(15 + 10 * random.random(), 1),
            # Simulación de humedad entre 50% y 80%
            'humidity': round(50 + 30 * random.random(), 1),
            # Simulación de precipitación entre 0 y 10 mm
            'precipitation': round(0 + 10 * random.random(), 1)
        }
        data.append(field_data)

    return JsonResponse(data, safe=False)


@user_required('scout')
def harvest_date_distribution(request):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    crops = Crop.objects.filter(field__scout=scout_id)
    harvest_dates = [crop.expected_harvest_date.strftime(
        '%Y-%m-%d') for crop in crops if crop.expected_harvest_date]

    date_counts = Counter(harvest_dates)
    sorted_dates = sorted(date_counts.items())

    data = {
        'dates': [date for date, count in sorted_dates],
        'counts': [count for date, count in sorted_dates]
    }

    return JsonResponse(data)


@user_required('scout')
def soil_type_distribution(request):
    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    # ✅ Obtener la distribución de tipos de suelo con el nombre del tipo de suelo
    soil_data = (
        Field.objects.filter(scout=scout_id)
        .values('soil_type__name')  # 🔹 Acceder al nombre del tipo de suelo
        .annotate(total_area=Sum('size'))
        .order_by('soil_type__name')  # 🔹 Ordenar por tipo de suelo
    )

    # ✅ Formatear los datos para el gráfico
    soil_types = [soil['soil_type__name']
                  # 🔹 Obtener nombres de los tipos de suelo
                  for soil in soil_data]
    # 🔹 Obtener el área total por tipo de suelo
    soil_areas = [soil['total_area'] for soil in soil_data]

    return JsonResponse({'soil_types': soil_types, 'soil_areas': soil_areas})


def crop_distribution_data(request, field_id):
    crops = Crop.objects.filter(field_id=field_id)
    crop_types = crops.values_list('crop_type', flat=True).distinct()

    data = []
    for crop_type in crop_types:
        count = crops.filter(crop_type=crop_type).count()
        data.append({'crop_type': crop_type, 'count': count})

    return JsonResponse(data, safe=False)


def field_size_comparison_view(request):
    fields = Field.objects.all()
    data = [{'field_name': field.name, 'size': field.size} for field in fields]
    return JsonResponse(data, safe=False)


def get_crop_types_data(request, field_id):
    field = get_object_or_404(Field, id=field_id)
    crops = field.crops.all()
    crop_types = crops.values('crop_type').annotate(count=Count('crop_type'))
    data = {
        'labels': [dict(Crop.CROP_TYPE_CHOICES).get(ctype['crop_type'], 'Desconocido') for ctype in crop_types],
        'counts': [ctype['count'] for ctype in crop_types]
    }
    return JsonResponse(data)


def get_growth_trend_data(request, field_id):
    field = get_object_or_404(Field, id=field_id)
    crops = field.crops.all()
    observations = CropObservation.objects.filter(
        crop__in=crops).order_by('date')
    dates = observations.values_list('date', flat=True).distinct()
    average_heights = [
        round(observations.filter(date=date).aggregate(
            Avg('crop_height'))['crop_height__avg'], 2)
        for date in dates
    ]
    data = {
        'dates': list(dates),
        'average_heights': average_heights
    }
    return JsonResponse(data)


def get_soil_climate_data(request, field_id):
    field = get_object_or_404(Field, id=field_id)
    crops = field.crops.all()
    observations = CropObservation.objects.filter(crop__in=crops)

    # Asegúrate de que realmente hay observaciones
    if not observations.exists():
        return JsonResponse({
            'soil_conditions': [],
            'heights': []
        })

    soil_conditions = observations.values(
        'date', 'soil_temperature', 'crop_height')
    data = {
        'dates': [obs['date'] for obs in soil_conditions],
        'soil_temperatures': [obs['soil_temperature'] for obs in soil_conditions],
        'crop_heights': [obs['crop_height'] for obs in soil_conditions]
    }
    return JsonResponse(data)


@user_required('scout')
def dashboard(request):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    # Gráficos de cultivos
    crops = Crop.objects.filter(field__scout=scout_id)

    # Agrupación mensual de alturas de cultivos
    monthly_data = crops.annotate(
        month=TruncMonth('observations__date')  # Agrupar por mes
    ).values('month').annotate(
        avg_height=Avg('observations__crop_height')  # Promediar la altura
    ).order_by('month')

    harvest_chart_data = {
        'dates': [data['month'].strftime('%Y-%m') for data in monthly_data if data['month'] is not None],
        'avg_heights': [data['avg_height'] for data in monthly_data if data['month'] is not None]
    }

    # Próximas cosechas
    today = datetime.today().date()
    upcoming_harvests = Crop.objects.filter(
        field__scout=scout_id,
        expected_harvest_date__gte=today,
        expected_harvest_date__lte=today + timedelta(days=30)
    ).order_by('expected_harvest_date')

    # Información del foro social
    recent_posts = Post.objects.all().order_by(
        '-created_at')[:5]  # Últimas 5 publicaciones

    # Recomendaciones (ajustada)
    recommendations = Recommendation.objects.filter(
        crop_observation__crop__field__scout=scout_id)

    # Noticias
    news_items = News.objects.all().order_by(
        '-created_at')[:5]  # Últimas 5 noticias

    # Recupera todos los campos asociados al Scout
    fields = Field.objects.filter(scout_id=scout_id)
    if not fields.exists():
        return render(request, 'dashboard.html', {'message': _('No hay campos asociados al scout.')})

    crops = Crop.objects.filter(field__in=fields)
    if not crops.exists():
        return render(request, 'dashboard.html', {'message': _('No hay cultivos en los campos del scout.')})

    projected_gains = []

    for crop in crops:
        num_plants = crop.num_plants or 0
        yield_per_plant = crop.yield_per_plant or 0
        market_price = crop.market_price or 0

        estimated_production = num_plants * yield_per_plant
        estimated_income = estimated_production * market_price

        projected_gains.append({
            'crop_name': crop.name,
            'num_plants': num_plants,
            'estimated_production': estimated_production,
            'market_price': market_price,
            'estimated_income': estimated_income,
        })

    # Comprobar si hay cultivos asociados a los campos del scout
    has_crops = Crop.objects.filter(field__scout=scout_id).exists()

    context = {
        'harvest_chart_data': harvest_chart_data,
        'upcoming_harvests': upcoming_harvests,
        'recent_posts': recent_posts,
        'recommendations': recommendations,
        'news_items': news_items,
        'projected_gains': projected_gains,
        'scout': scout_id,
        'has_crops': has_crops,
    }

    return render(request, 'dashboard.html', context)


@user_required('scout')
def crop_growth_chart(request):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    # Crear un diccionario para almacenar el tamaño acumulado de las plantas por cultivo y por mes
    growth_data = defaultdict(lambda: defaultdict(float))

    # Obtener todas las observaciones de los cultivos del Scout actual
    observations = CropObservation.objects.filter(crop__field__scout=scout_id)

    # Organizar los datos por cultivo y por mes
    for observation in observations:
        crop_name = observation.crop.name
        month = observation.date.strftime('%Y-%m')
        growth_data[crop_name][month] += observation.crop_height

    # Preparar los datos para el gráfico
    series = []
    categories = sorted(set(month for crop_data in growth_data.values()
                        for month in crop_data.keys()))

    for crop_name, crop_data in growth_data.items():
        series.append({
            'name': crop_name,
            'data': [crop_data.get(month, 0) for month in categories]
        })

    # Verifica el contenido de la respuesta antes de enviarla
    if not series or not categories:
        return JsonResponse({'error': _('No hay datos disponibles')}, status=404)

    return JsonResponse({'categories': categories, 'series': series})


def dashboard_view(request):
    return render(request, 'dashboard.html')


@user_required('scout')
def recommendations_view(request):

    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    # Filtrar recomendaciones basadas en las observaciones de cultivos del scout actual
    recommendations = Recommendation.objects.filter(
        crop_observation__crop__field__scout=scout_id)

    context = {
        'recommendations': recommendations
    }

    return render(request, 'recommendations.html', context)


@user_required('scout')
def mark_all_read(request):
    scout_id = request.session.get('scout_id')
    if scout_id:
        Notification.objects.filter(
            scout_id=scout_id, status='unread').update(status='read')
        # Redirigir a la página anterior si está disponible
        previous_url = request.META.get('HTTP_REFERER', '/')
        return redirect(previous_url)
    return redirect('/')  # Redirige al inicio si no hay scout_id


@user_required('scout')
def edit_scout_profile(request, scout_id):

    scout = get_current_scout(request)
    scout = Scout.objects.get(id=scout_id)

    if request.method == 'POST':
        form = ScoutProfileForm(request.POST, request.FILES, instance=scout)
        if form.is_valid():
            form.save()
            messages.success(request, _('Perfil actualizado con éxito.'))
            return redirect('edit_scout_profile', scout_id=scout.id)
    else:
        form = ScoutProfileForm(instance=scout)

    return render(request, 'edit_scout_profile.html', {'form': form})


def get_plant_count_history(request, crop_id):
    crop = get_object_or_404(Crop, id=crop_id)

    # Obtener todas las observaciones del cultivo, ordenadas por fecha
    observations = CropObservation.objects.filter(crop=crop).order_by('date')

    # Preparar los datos para el gráfico
    data = {
        # Fechas de observación
        "dates": [obs.date.strftime("%Y-%m-%d") for obs in observations],
        # Cantidad de plantas en cada observación
        "plant_counts": [obs.stand_count for obs in observations]
    }

    return JsonResponse(data)


@user_required('scout')
def create_crop(request):
    scout_id = request.session.get('scout_id')
    if not scout_id:
        raise ValueError("El ID del Scout no se encuentra en la sesión")

    if request.method == 'POST':
        form = CropForm(request.POST)
        if form.is_valid():
            # Crear la instancia pero no guardar aún
            crop = form.save(commit=False)

            # Capturar los valores de nutrientes
            nitrogen = request.POST.get('nitrogen')
            phosphorus = request.POST.get('phosphorus')
            potassium = request.POST.get('potassium')

            # Convertirlos a JSON
            crop.nutrient_requirements = {
                "N": int(nitrogen),
                "P": int(phosphorus),
                "K": int(potassium)
            }

            crop.scout = scout_id  # Asignar el scout actual si es necesario
            crop.save()  # Guardar el cultivo
            messages.success(request, _("Cultivo creado con éxito."))
            return redirect('crop-list')
    else:
        form = CropForm()

    # Proveer campos al formulario si se necesita
    fields = Field.objects.filter(scout_id=scout_id)
    return render(request, 'create_crop.html', {'form': form, 'fields': fields})


logging.basicConfig(level=logging.DEBUG)


# Inicializa el cliente con tu API key y la URL de la API
CLIENT = InferenceHTTPClient(
    api_url="https://detect.roboflow.com",
    api_key="9yQRQUi4kEsP5XKsucss"
)


@csrf_exempt
def count_plants(request):
    if request.method == "POST":
        try:
            # Cargar JSON desde request.body
            data = json.loads(request.body)

            # Extraer la imagen en base64
            image_data = data.get("image", "")
            if "," in image_data:
                image_data = image_data.split(",")[1]

            # Decodificar base64 a bytes
            image_bytes = base64.b64decode(image_data)

            # Guardar temporalmente la imagen en disco
            with tempfile.NamedTemporaryFile(suffix=".jpg") as tmp:
                tmp.write(image_bytes)
                tmp.flush()

                # Realizar la inferencia con el modelo de Roboflow
                result = CLIENT.infer(
                    tmp.name, model_id="plant-detection-iqmnt/1")

            # Procesar el resultado de la inferencia
            predictions = result.get("predictions", [])
            num_plants = len(
                [pred for pred in predictions if pred.get("class") == "plant"])

            return JsonResponse({"num_plants": num_plants})

        except Exception as e:
            print(f"🚨 Error en conteo de plantas: {str(e)}")
            return JsonResponse({"error": "Error al procesar la imagen"}, status=500)

    return JsonResponse({"error": "Método no permitido"}, status=405)


# Cargar el modelo YOLO previamente descargado


# @csrf_exempt
# def count_plants(request):
#     model = YOLO("yolov8n.pt")  # Asegúrate de tener el modelo correcto
#     #torch.backends.nnpack.enabled = False

#     if request.method == "POST":
#         try:
#             # Cargar JSON desde request.body
#             data = json.loads(request.body)

#             # Extraer la imagen base64
#             image_data = data.get("image", "").split(",")[1]  # Omitir "data:image/png;base64,"

#             # Decodificar base64 a bytes
#             image_bytes = base64.b64decode(image_data)

#             # Convertir bytes en imagen OpenCV
#             image_array = np.frombuffer(image_bytes, dtype=np.uint8)
#             image = cv2.imdecode(image_array, cv2.IMREAD_COLOR)

#             # Procesar imagen con YOLO
#             results = model(image)
#             num_plants = len(results[0].boxes)  # Contar objetos detectados

#             return JsonResponse({"num_plants": num_plants})

#         except Exception as e:
#             print(f"🚨 Error en conteo de plantas: {str(e)}")
#             return JsonResponse({"error": "Error al procesar la imagen"}, status=500)

#     return JsonResponse({"error": "Método no permitido"}, status=405)


# @csrf_exempt  # Solo para pruebas, en producción usa @csrf_protect
# def count_plants_video(request):
#     if request.method == "POST" and request.FILES.get("video"):
#         video_file = request.FILES["video"]

#         # Convertir video en un array de OpenCV
#         video_bytes = video_file.read()
#         video_array = np.frombuffer(video_bytes, np.uint8)

#         # Guardar el video temporalmente
#         temp_video_path = "/tmp/temp_video.mp4"
#         with open(temp_video_path, "wb") as f:
#             f.write(video_bytes)

#         # Abrir el video con OpenCV
#         cap = cv2.VideoCapture(temp_video_path)

#         total_plants = 0
#         frame_count = 0

#         while cap.isOpened():
#             ret, frame = cap.read()
#             if not ret:
#                 break  # Fin del video

#             # Procesar con YOLO cada 10 cuadros para optimizar rendimiento
#             if frame_count % 10 == 0:
#                 results = model(frame)
#                 num_plants = len(results[0].boxes)  # Contar objetos detectados
#                 total_plants += num_plants  # Acumular conteo

#             frame_count += 1

#         cap.release()

#         return JsonResponse({"num_plants": total_plants})

#     return JsonResponse({"error": "Método no permitido"}, status=405)

def get_subscriber_hash(email):
    """Calcular el hash MD5 del email para Mailchimp."""
    return hashlib.md5(email.lower().encode('utf-8')).hexdigest()


def subscribe_mailchimp(request):
    if request.method == 'POST':
        email = request.POST.get('email')
        if not email:
            messages.error(request, _(
                "Por favor, ingresa un correo electrónico válido."))
            return redirect('landing-page')  # Página de regreso

        try:
            # Configurar Mailchimp Client
            client = Client()
            client.set_config({
                "api_key": settings.MAILCHIMP_API_KEY,
                "server": settings.MAILCHIMP_SERVER_PREFIX
            })

            # Calcular el hash del suscriptor
            subscriber_hash = get_subscriber_hash(email)

            # Añadir o actualizar miembro en la lista
            client.lists.set_list_member(
                settings.MAILCHIMP_LIST_ID,
                subscriber_hash,
                {
                    "email_address": email,
                    "status_if_new": "subscribed",
                }
            )

            # Agregar etiqueta "waiting-list"
            client.lists.update_list_member_tags(
                settings.MAILCHIMP_LIST_ID,
                subscriber_hash,
                body={
                    "tags": [{"name": "waiting-list", "status": "active"}]
                }
            )

            # Mostrar mensaje de éxito
            messages.success(request, _(
                "¡Te has registrado exitosamente en la lista de espera!"))
        except ApiClientError as error:
            logger.error(f"Error en Mailchimp: {error.text}")

            if 'Member Exists' in error.text:
                messages.info(request, _(
                    "Este correo ya está registrado en la lista de espera."))
            else:
                messages.error(request, _(
                    "Hubo un error al procesar tu solicitud. Inténtalo de nuevo más tarde."))

        # Redirigir a la página principal
        return redirect('landing-page')

    return redirect('landing-page')  # Redirigir si no es POST


@user_required('scout')
def soil_test_list(request, field_id):
    """Muestra el historial de análisis de suelo de un campo para Scouts."""
    field = get_object_or_404(
        Field, id=field_id, scout=request.session.get('scout_id'))
    soil_tests = SoilTest.objects.filter(field=field).order_by('-date')

    form = SoilTestForm()

    return render(request, 'soil_test_list.html', {
        'field': field,
        'soil_tests': soil_tests,
        'form': form,
    })


@user_required('scout')
def add_soil_test(request, field_id):
    """Permite a un Scout registrar un nuevo análisis de suelo desde un modal."""
    field = get_object_or_404(
        Field, id=field_id, scout=request.session.get('scout_id'))

    if request.method == 'POST':
        form = SoilTestForm(request.POST)
        if form.is_valid():
            soil_test = form.save(commit=False)
            soil_test.field = field
            soil_test.save()

            # Generar recomendaciones automáticas
            generate_soil_recommendations(soil_test)

            # ✅ Redirigir al listado de análisis de suelo después de guardar
            return redirect('scout-soil-test-list', field_id=field.id)

    else:
        form = SoilTestForm()

    return render(request, 'soil_test_list.html', {'field': field, 'form': form})



#SOCIAL NETWORK

@user_required('scout')
def post_list(request):
    scout_id = request.session.get('scout_id')
    if not scout_id:
        return redirect('login')  # Redirigir al login si no está autenticado

    # Obtener el ContentType para Scout
    scout_content_type = ContentType.objects.get_for_model(Scout)

    # Obtener los posts del usuario actual
    posts = Post.objects.filter(author_type=scout_content_type, author_id=scout_id).order_by('-created_at').prefetch_related('comments')

    # Pasar el ContentType y el ID del usuario a la plantilla
    return render(request, 'post_list.html', {
        'posts': posts,
        'scout_content_type': scout_content_type,
        'scout_id': scout_id,
    })

@user_required('scout')
def create_post(request):
    scout_id = request.session.get('scout_id')
    scout_content_type = ContentType.objects.get_for_model(Scout)

    if request.method == 'POST':
        form = PostForm(request.POST)
        if form.is_valid():
            post = form.save(commit=False)
            post.author_type = scout_content_type
            post.author_id = scout_id
            post.save()  # Guardar el post en la base de datos
            return redirect('post_list')
    else:
        form = PostForm()

    posts = Post.objects.filter(author_type=scout_content_type, author_id=scout_id)
    return render(request, 'post_list.html', {'form': form, 'posts': posts})


@user_required('scout')
def edit_post(request, post_id):
    scout_id = request.session.get('scout_id')
    scout_content_type = ContentType.objects.get_for_model(Scout)

    # Obtener el post solo si el autor es el usuario actual
    post = get_object_or_404(Post, id=post_id, author_type=scout_content_type, author_id=scout_id)

    if request.method == 'POST':
        form = PostForm(request.POST, instance=post)
        if form.is_valid():
            form.save()
            return redirect('post_list')  # Redirigir a la lista de posts
    else:
        form = PostForm(instance=post)

    # Mostrar el formulario de edición en la misma página de lista de posts
    posts = Post.objects.filter(author_type=scout_content_type, author_id=scout_id)
    return render(request, 'post_list.html', {'form': form, 'posts': posts, 'editing_post': post})    

@user_required('scout')  # Asegúrate de que el usuario esté autenticado como Scout
def delete_post(request, post_id):
    # Obtener el ID del usuario actual desde la sesión
    scout_id = request.session.get('scout_id')
    if not scout_id:
        messages.error(request, "No estás autenticado.")
        return redirect('login')  # Redirigir al login si no está autenticado

    # Obtener el ContentType para Scout
    scout_content_type = ContentType.objects.get_for_model(Scout)

    # Obtener el post solo si el autor es el usuario actual
    post = get_object_or_404(Post, id=post_id, author_type=scout_content_type, author_id=scout_id)

    if request.method == 'POST':
        # Eliminar el post
        post.delete()
        messages.success(request, "La publicación se ha eliminado correctamente.")
        return redirect('post_list')  # Redirigir a la lista de posts

    # Si no es una solicitud POST, redirigir a la lista de posts
    return redirect('post_list')

@user_required('scout')  # Asegúrate de que el usuario esté autenticado como Scout
def add_comment(request, post_id):
    # Obtener el ID del usuario actual desde la sesión
    scout_id = request.session.get('scout_id')
    if not scout_id:
        messages.error(request, "No estás autenticado.")
        return redirect('login')  # Redirigir al login si no está autenticado

    # Obtener el ContentType para Scout
    scout_content_type = ContentType.objects.get_for_model(Scout)

    # Obtener el post al que se agregará el comentario
    post = get_object_or_404(Post, id=post_id)

    if request.method == 'POST':
        form = CommentForm(request.POST)
        if form.is_valid():
            comment = form.save(commit=False)
            comment.post = post
            comment.author_type = scout_content_type
            comment.author_id = scout_id
            comment.save()
            messages.success(request, "El comentario se ha agregado correctamente.")
            return redirect('post_list')  # Redirigir a la lista de posts
    else:
        form = CommentForm()

    # Si no es una solicitud POST, redirigir a la lista de posts
    return redirect('post_list')


@user_required('scout')  # Asegúrate de que el usuario esté autenticado como Scout
def edit_comment(request, comment_id):
    # Obtener el ID del usuario actual desde la sesión
    scout_id = request.session.get('scout_id')
    if not scout_id:
        messages.error(request, "No estás autenticado.")
        return redirect('login')  # Redirigir al login si no está autenticado

    # Obtener el ContentType para Scout
    scout_content_type = ContentType.objects.get_for_model(Scout)

    # Obtener el comentario solo si el autor es el usuario actual
    comment = get_object_or_404(Comment, id=comment_id, author_type=scout_content_type, author_id=scout_id)

    if request.method == 'POST':
        form = CommentForm(request.POST, instance=comment)
        if form.is_valid():
            form.save()
            messages.success(request, "El comentario se ha editado correctamente.")
            return redirect('post_list')  # Redirigir a la lista de posts
    else:
        form = CommentForm(instance=comment)

    # Pasar el formulario a la plantilla
    posts = Post.objects.filter(author_type=scout_content_type, author_id=scout_id)
    return render(request, 'post_list.html', {
        'form': form,
        'posts': posts,
        'editing_comment': comment,  # Para identificar que estamos editando un comentario
    })


@user_required('scout')  # Asegúrate de que el usuario esté autenticado como Scout
def delete_comment(request, comment_id):
    # Obtener el ID del usuario actual desde la sesión
    scout_id = request.session.get('scout_id')
    if not scout_id:
        messages.error(request, "No estás autenticado.")
        return redirect('login')  # Redirigir al login si no está autenticado

    # Obtener el ContentType para Scout
    scout_content_type = ContentType.objects.get_for_model(Scout)

    # Obtener el comentario solo si el autor es el usuario actual
    comment = get_object_or_404(Comment, id=comment_id, author_type=scout_content_type, author_id=scout_id)

    if request.method == 'POST':
        comment.delete()
        messages.success(request, "El comentario se ha eliminado correctamente.")
        return redirect('post_list')  # Redirigir a la lista de posts

    # Si no es una solicitud POST, redirigir a la lista de posts
    return redirect('post_list')

def recursos_scouts(request):
    books = Book.objects.all().order_by('-uploaded_at')
    videos = Video.objects.all().order_by('-uploaded_at')
    return render(request, 'recursos_scouts.html', {
        'books': books,
        'videos': videos
    })


def agriculture_course_view(request):
    return render(request, 'curso-agro.html', {
        'title': 'Curso de Agricultura de Precisión y Sostenible'
    })