import requests
from django.utils import timezone  # Import requerido
from datetime import timedelta  # Para calcular diferencias de fechas

def get_current_scout(request):
    from .models import Scout

    # Asumiendo que 'scout_id' se guarda en la sesión durante el login
    scout_id = request.session.get('scout_id')

    if scout_id:
        try:
            return Scout.objects.get(id=scout_id)
        except Scout.DoesNotExist:
            return None
    return None


def calculate_nitrogen(soil_test, yield_goal):
    """Calcula la cantidad de nitrógeno necesaria en función del análisis de suelo y el rendimiento objetivo."""
    nitrogen_coefficient = 2.5  # Según las matrices
    nitrogen_needed = max(
        0, (yield_goal * nitrogen_coefficient) - soil_test.nitrate_ppm)
    return nitrogen_needed


def calculate_phosphorus(soil_test, yield_goal):
    """Calcula la cantidad de fósforo necesaria en función del análisis de suelo y el rendimiento objetivo."""
    soil_test_p = soil_test.phosphorus_ppm
    if soil_test_p <= 8:
        return yield_goal * 80
    elif 9 <= soil_test_p <= 16:
        return yield_goal * 60
    elif 17 <= soil_test_p <= 35:
        return yield_goal * 40
    elif 36 <= soil_test_p <= 50:
        return 0
    else:
        return 0


def calculate_potassium(soil_test, yield_goal):
    """Calcula la cantidad de potasio necesaria en función del análisis de suelo y el rendimiento objetivo."""
    soil_test_k = soil_test.potassium_ppm
    if soil_test_k < 61:
        return yield_goal * 100
    elif 61 <= soil_test_k <= 90:
        return yield_goal * 80
    elif 91 <= soil_test_k <= 130:
        return yield_goal * 60
    elif 131 <= soil_test_k <= 175:
        return yield_goal * 40
    else:
        return 0


# def generate_recommendations(soil_test, crop, crop_observation):
#     from .models import Recommendation, Brand

#     if not soil_test or not crop or not crop_observation:
#         raise ValueError("Faltan datos para generar la recomendación.")

#     yield_goal = soil_test.yield_target
#     nitrogen = calculate_nitrogen(soil_test, yield_goal)
#     phosphorus = calculate_phosphorus(soil_test, yield_goal)
#     potassium = calculate_potassium(soil_test, yield_goal)

#     description = f"Aplicar: {nitrogen:.2f} lbs de N, {phosphorus:.2f} lbs de P2O5, {potassium:.2f} lbs de K2O."

#     # Buscar productos recomendados
#     nitrogen_product = Brand.objects.filter(product_type='nitrogen', availability=True).first()
#     phosphorus_product = Brand.objects.filter(product_type='phosphorus', availability=True).first()
#     potassium_product = Brand.objects.filter(product_type='potassium', availability=True).first()

#     if nitrogen_product:
#         description += f"\nRecomendación N: {nitrogen_product.name} ({nitrogen_product.price_per_unit} por {nitrogen_product.unit_of_measure})."
#     if phosphorus_product:
#         description += f"\nRecomendación P: {phosphorus_product.name} ({phosphorus_product.price_per_unit} por {phosphorus_product.unit_of_measure})."
#     if potassium_product:
#         description += f"\nRecomendación K: {potassium_product.name} ({potassium_product.price_per_unit} por {potassium_product.unit_of_measure})."

#     # Enviar recomendación a un Webhook para procesarla con IA
#     webhook_url = "https://hook.us2.make.com/wo4yyvald9tfrg2g5yrm2jttwrz51re7"  # ✅ Cambia por tu URL real
#     payload = {
#         "crop": crop.name,
#         "soil_test_date": str(soil_test.date),
#         "recommendation": description
#     }

#     try:
#         response = requests.post(webhook_url, json=payload, timeout=10)
#         if response.status_code == 200:
#             # ✅ Decodificar la respuesta correctamente
#             ia_response = response.text.encode('latin1').decode('utf-8')
#             description = ia_response  # ✅ Guardar la versión procesada por la IA
#         else:
#             print(f"Error en el webhook, código {response.status_code}")
#     except requests.exceptions.RequestException as e:
#         print(f"Error al conectar con el webhook: {e}")

#     # Guardar la recomendación en la base de datos con la versión mejorada
#     Recommendation.objects.create(
#         crop=crop,
#         soil_test=soil_test,
#         crop_observation=crop_observation,
#         nitrogen_lbs_per_acre=nitrogen,
#         phosphorus_lbs_per_acre=phosphorus,
#         potassium_lbs_per_acre=potassium,
#         description=description,  # ✅ Guarda la recomendación procesada por IA
#         title=f"Recomendación para {crop.name}",
#     )


def generate_recommendations(soil_test, crop, crop_observation):
    """Genera recomendaciones para cultivos basadas en `SoilTest` y las envía a la IA para mejorar el texto."""
    WEBHOOK_URL = "https://hook.us2.make.com/ejutio927rnst1y3156a4iisk37igiki"  # URL del Webhook

    from .models import Recommendation, Brand

    # Verificar que todos los datos sean válidos antes de continuar
    if not soil_test or not crop or not crop_observation:
        print("⚠️ ERROR: Faltan datos para generar la recomendación.")
        return

    yield_goal = soil_test.yield_target if soil_test.yield_target else 0
    nitrogen = calculate_nitrogen(soil_test, yield_goal)
    phosphorus = calculate_phosphorus(soil_test, yield_goal)
    potassium = calculate_potassium(soil_test, yield_goal)


    # Input narrativo estructurado
    input_text = f"""
    ## Contexto Agronómico
    Cultivo: {crop.name}
    Fecha de siembra: {crop.planting_date.strftime('%Y-%m-%d')}
    Días desde siembra: {(timezone.now().date() - crop.planting_date).days} días
    
    ## Análisis de Suelo (Fecha: {soil_test.date.strftime('%Y-%m-%d')})
    - pH: {soil_test.pH}
    - Materia Orgánica: {soil_test.organic_matter_percentage}%
    - Nitrógeno (N): {soil_test.nitrate_ppm} ppm
    - Fósforo (P): {soil_test.phosphorus_ppm} ppm
    - Potasio (K): {soil_test.potassium_ppm} ppm
    
    ## Observación de Campo (Fecha: {crop_observation.date.strftime('%Y-%m-%d')})
    - Estado del cultivo: 
      * Altura: {crop_observation.crop_height} cm
      * Densidad: {crop_observation.stand_count} plantas/m²
    - Malezas: 
      * Tipo: {crop_observation.weed_pressure}
      * Altura: {crop_observation.weed_height} cm
    - Plagas: {crop_observation.pests if crop_observation.pests else 'No reportadas'}
    - Lluvia acumulada: {crop_observation.rain_accumulation} mm
    - Temperatura suelo: {crop_observation.soil_temperature}°C
    
    ## Requerimientos Nutricionales Calculados
    - Nitrógeno (N): {nitrogen:.2f} lbs/acre
    - Fósforo (P₂O₅): {phosphorus:.2f} lbs/acre
    - Potasio (K₂O): {potassium:.2f} lbs/acre
    
    Tarea: Genera una recomendación integrada considerando:
    1. Balance nutricional
    2. Estado fenológico del cultivo
    3. Presencia de malezas/plagas
    4. Condiciones ambientales recientes
    """

    # ✅ Datos que se enviarán al Webhook
    payload = {
        "crop_id": crop.id,
        "observation_id": crop_observation.id,
        "input": input_text.strip(),
        "strict_mode": "single_recommendation"
    }

    try:
        # ⏳ Aumentar timeout a 20 seg
        response = requests.post(WEBHOOK_URL, json=payload, timeout=30)

        if response.status_code != 200:
            print(
                f"⚠️ ERROR: Webhook respondió con código {response.status_code}. No se generó la recomendación.")
            return

        # ✅ Obtener la respuesta del Webhook como texto plano
        ia_response = response.text.strip()

        # 🚨 Manejar respuestas vacías o incorrectas
        if not ia_response or ia_response.lower() == "accepted":
            print(f"⚠️ ERROR: Respuesta inesperada de la IA: {ia_response}")
            return

        print(f"✅ Respuesta recibida de la IA: {ia_response}")

        # ✅ Verificar si ya existe una recomendación para este cultivo y análisis de suelo
        if Recommendation.objects.filter(crop=crop, soil_test=soil_test, crop_observation=crop_observation).exists():
            print(
                f"⚠️ Ya existe una recomendación para el cultivo '{crop.name}' y el análisis de suelo {soil_test.id}. No se generará otra.")
            return

        # ✅ Crear la recomendación en la base de datos (sin los campos problemáticos)
        Recommendation.objects.create(
            crop=crop,
            soil_test=soil_test,
            crop_observation=crop_observation,
            description=ia_response,  # ✅ Guarda la recomendación procesada por IA
            title=f"Recomendación para {crop.name}",
        )

        print(
            f"✅ Recomendación generada y guardada para el cultivo '{crop.name}'")

    except requests.exceptions.Timeout:
        print(f"⚠️ ERROR: Tiempo de espera agotado al conectar con el webhook.")
    except requests.exceptions.RequestException as e:
        print(f"⚠️ ERROR al conectar con la IA: {e}")





def generate_soil_recommendations(soil_test):
    # 🔹 URL del Webhook para la IA
    WEBHOOK_URL = "https://hook.us2.make.com/s0ug3lp8c9t4y88vea4knp4q1fqasku1"

    """Genera recomendaciones de suelo basadas en `SoilTest`, enviando un input más detallado a la IA."""

    from .models import Recommendation

    field = soil_test.field

    if not field:
        print(f"⚠️ ERROR: El análisis de suelo '{soil_test.id}' no tiene un campo asociado.")
        return

    print(f"✅ Enviando datos del análisis de suelo del campo '{field.name}' al Webhook...")

    # Datos básicos
    yield_goal = soil_test.yield_target if soil_test.yield_target else 0
    nitrogen_needed = calculate_nitrogen(soil_test, yield_goal)
    phosphorus_needed = calculate_phosphorus(soil_test, yield_goal)
    potassium_needed = calculate_potassium(soil_test, yield_goal)

    # 🚀 Creamos el input de manera narrativa
    input_text = f"""
    Campo: {field.name}
    Fecha del análisis: {soil_test.date.strftime('%Y-%m-%d')}
    Resultados:
    - pH del suelo: {soil_test.pH}
    - Materia Orgánica: {soil_test.organic_matter_percentage}%
    - Nitrógeno disponible: {soil_test.nitrate_ppm} ppm
    - Fósforo disponible: {soil_test.phosphorus_ppm} ppm
    - Potasio disponible: {soil_test.potassium_ppm} ppm

    Requerimientos estimados:
    - Nitrógeno: {nitrogen_needed} lbs/acre
    - Fósforo: {phosphorus_needed} lbs/acre
    - Potasio: {potassium_needed} lbs/acre

    Condiciones actuales:
    - Clima: (Dato no disponible actualmente)
    - Disponibilidad de agua: (Dato no disponible actualmente)
    - Tipo de suelo: (Dato no disponible actualmente)

    Tarea: Basado en estos datos, genera una recomendación concreta para mejorar la salud del suelo y maximizar el rendimiento del campo, sin repetir prácticas previas que a continuación se darán.
        """.strip()

    # Webhook de Make.com
    payload = {
        "field_id": field.id,
        "input": input_text
    }

    try:
        response = requests.post(WEBHOOK_URL, json=payload, timeout=30)

        if response.status_code != 200:
            print(f"⚠️ ERROR: Webhook respondió con código {response.status_code}. No se generó la recomendación.")
            return

        recommendation_text = response.text.strip()

        # 🚨 Verificar si la respuesta de la IA es válida
        if not recommendation_text or recommendation_text.lower() == "accepted":
            print(f"⚠️ ERROR: Respuesta inesperada de la IA: {recommendation_text}")
            return

        print(f"✅ Respuesta recibida de la IA: {recommendation_text}")

        if Recommendation.objects.filter(field=field, soil_test=soil_test).exists():
            print(f"⚠️ Ya existe una recomendación para el campo '{field.name}' y el análisis de suelo {soil_test.id}. No se generará otra.")
            return

        Recommendation.objects.create(
            field=field,
            soil_test=soil_test,
            title=f"Recomendaciones de suelo para el campo '{field.name}'",
            description=recommendation_text,
        )

        print(f"✅ Recomendación generada y guardada para el campo '{field.name}'")

    except requests.RequestException as e:
        print(f"⚠️ ERROR al conectar con la IA: {e}")
