from django.db.models.signals import post_save
from django.dispatch import receiver
from django.core.mail import send_mail
from django.conf import settings
from scouts.models import Recommendation, SoilTest
from scouts.models import Notification
from django.template.loader import render_to_string
from django.utils.html import strip_tags
import logging

from .utils import generate_soil_recommendations

logger = logging.getLogger(__name__)


@receiver(post_save, sender=Recommendation)
def create_recommendation_notification(sender, instance, created, **kwargs):
    """
    Genera una notificación y envía un correo cuando se crea una recomendación.
    Ahora, Scouts y Farmers recibirán TODAS las notificaciones, sin importar si es un Campo o un Cultivo.
    """
    if created:
        try:
            # Determinar si la recomendación es para un Cultivo o un Campo
            recipient_type = "cultivo" if instance.crop else "campo"
            recipient_name = instance.crop.name if instance.crop else instance.field.name

            # Obtener el Farmer
            farmer = instance.crop.field.farmer if instance.crop else instance.field.farmer

            # Obtener el Scout si la recomendación es para un Cultivo
            scout = instance.crop.field.scout if instance.crop else None

            # 🔹 Notificación para el Scout (ahora también recibe notificaciones de Campos)
            if scout:
                Notification.objects.create(
                    scout=scout,
                    message=f"Se ha generado una nueva recomendación para el {recipient_type} {recipient_name}: {instance.title}.",
                    type='fertilization',
                    status='unread',
                )
                logger.info(
                    f"Notificación creada para el scout {scout.first_name} {scout.last_name}.")

                # Enviar correo al Scout
                send_recommendation_email(
                    scout.email, scout.first_name, recipient_name, instance)

            # 🔹 Notificación para el Farmer (ahora también recibe notificaciones de Cultivos)
            if farmer:
                Notification.objects.create(
                    farmer=farmer,
                    message=f"Se ha generado una nueva recomendación para el {recipient_type} {recipient_name}: {instance.title}.",
                    type='fertilization',
                    status='unread',
                )
                logger.info(
                    f"Notificación creada para el farmer {farmer.first_name} {farmer.last_name}.")

                # Enviar correo al Farmer
                send_recommendation_email(
                    farmer.email, farmer.first_name, recipient_name, instance)

        except AttributeError as e:
            logger.error(f"Error al acceder a los datos relacionados: {e}")
        except Exception as e:
            logger.error(f"Error inesperado al crear la notificación: {e}")


def send_recommendation_email(email, name, recipient_name, instance):
    """
    Envía un correo electrónico con la recomendación generada.
    """

    if email:
        try:
            email_subject = "Nueva Recomendación en AgriCarm"
            email_context = {
                "name": name,
                "crop_name": instance.crop.name if instance.crop else None,
                "field_name": instance.field.name if instance.field else None,
                "recommendation_title": instance.title,
                "recommendation_description": instance.description,
            }

            email_html_content = render_to_string(
                "email/notification-email.html", email_context)
            email_plain_content = strip_tags(email_html_content)

            send_mail(
                subject=email_subject,
                message=email_plain_content,
                from_email="AgriCarm App <no-reply@agricarm.com>",
                recipient_list=[email],
                html_message=email_html_content
            )

            logger.info(f"Correo enviado exitosamente a {email}.")
        except Exception as e:
            logger.error(f"Error al enviar el correo a {email}: {e}")
    else:
        logger.warning(
            f"El usuario {name} no tiene un correo electrónico definido.")


@receiver(post_save, sender=SoilTest)
def generate_recommendations_on_soil_test(sender, instance, **kwargs):
    """
    Genera recomendaciones automáticamente cuando se guarda un análisis de suelo.
    """
    generate_soil_recommendations(
        instance)  # ✅ Solo pasamos `instance`, que es el `soil_test`
