# Librerías estándar de Python
from datetime import date, timedelta

# Importaciones de Django
from django.db import models
from django.contrib.auth.models import User
from django.db.models.signals import post_save
from django.dispatch import receiver
from django.utils import timezone
from django.conf import settings
from django.contrib.contenttypes.models import ContentType
from django.contrib.contenttypes.fields import GenericForeignKey
from django.utils.translation import gettext_lazy as _
from django.db import models


# Importaciones locales
from .utils import generate_recommendations


# Create your models here.
class Scout(models.Model):
    first_name = models.CharField(max_length=100)
    last_name = models.CharField(max_length=100)
    email = models.EmailField(unique=True)
    phone = models.CharField(max_length=15, blank=True, null=True)
    address = models.CharField(max_length=255, blank=True, null=True)
    user_type = models.CharField(max_length=50, choices=[(
        'scout', 'Scout'), ('agronomist', 'Agronomist')])
    profile_picture = models.ImageField(
        upload_to='profile_pictures/', blank=True, null=True)
    farmer = models.ForeignKey('farmer.Farmer', on_delete=models.SET_NULL,
                               related_name='assigned_scouts', null=True, blank=True)

    def __str__(self):
        return f"{self.first_name} {self.last_name}"


class Token(models.Model):
    # Relación genérica para vincular el token a cualquier modelo de usuario
    content_type = models.ForeignKey(
        ContentType, on_delete=models.CASCADE, null=True, blank=True)
    object_id = models.PositiveIntegerField(null=True, blank=True)
    user = GenericForeignKey('content_type', 'object_id')

    token = models.CharField(max_length=6)
    created_at = models.DateTimeField(auto_now_add=True)
    expires_at = models.DateTimeField()

    def is_valid(self):
        """Verifica si el token sigue siendo válido."""
        return self.expires_at >= timezone.now()

    def save(self, *args, **kwargs):
        """Asigna una fecha de expiración predeterminada si no está configurada."""
        if not self.expires_at:
            # Expira en 10 minutos por defecto
            self.expires_at = timezone.now() + timedelta(minutes=10)
        super().save(*args, **kwargs)

    class Meta:
        verbose_name = "Token"
        verbose_name_plural = "Tokens"
        ordering = ['-created_at']

    def __str__(self):
        return f"Token {self.token} for {self.user} (Expires at: {self.expires_at})"


# 🔹 MODELO FIELD (CAMPOS)

class SoilType(models.Model):
    name = models.CharField(max_length=100, unique=True)
    description = models.TextField()

    def translated_description(self):
        """
        Devuelve la descripción traducida si está disponible en el archivo de traducción.
        """
        return _(self.description)

    def __str__(self):
        return self.name


class Field(models.Model):

    name = models.CharField(max_length=200)
    location = models.CharField(max_length=200, null=True, blank=True)
    latitude = models.FloatField()
    longitude = models.FloatField()
    size = models.FloatField(help_text=_(
        "Tamaño del campo en acres o hectáreas"))

    # 🔹 Modificación: soil_type ahora es una ForeignKey
    soil_type = models.ForeignKey(
        SoilType, on_delete=models.SET_NULL, null=True, blank=True)

    # Dueño del campo
    farmer = models.ForeignKey(
        'farmer.Farmer', on_delete=models.CASCADE, related_name="fields")
    # Scout que lo gestiona (opcional)
    scout = models.ForeignKey('scouts.Scout', on_delete=models.SET_NULL,
                              null=True, blank=True, related_name="fields_managed")

    def __str__(self):
        return f"{self.name} ({self.location})"


class SoilTest(models.Model):
    field = models.ForeignKey('Field', on_delete=models.CASCADE,
                              related_name='soil_tests', null=True, blank=True)
    date = models.DateField()
    pH = models.FloatField(null=True, blank=True)  # ✅ Se agrega el campo pH
    nitrate_ppm = models.FloatField()  # Nitrato en ppm
    phosphorus_ppm = models.FloatField()  # Fósforo en ppm
    potassium_ppm = models.FloatField()  # Potasio en ppm
    zinc_ppm = models.FloatField(null=True, blank=True)  # Zinc en ppm
    sulfur_ppm = models.FloatField(null=True, blank=True)  # Azufre en ppm
    organic_matter_percentage = models.FloatField()  # Materia orgánica en porcentaje
    bulk_density = models.FloatField()  # Densidad aparente en g/cm³
    yield_target = models.FloatField()  # Rendimiento objetivo en bushels/acre

    def __str__(self):
        return f"Soil Test on {self.date} for {self.field.name}"


# 🔹 MODELO CROP (CULTIVOS)
class Crop(models.Model):
    CROP_TYPE_CHOICES = [
        ('FR', _('Frutas')),
        ('VG', _('Vegetales')),
        ('GR', _('Granos')),
        ('LG', _('Legumbres')),
        ('CR', _('Cereal')),
        ('NT', _('Nueces')),
        ('SP', _('Especias')),
        ('HR', _('Hierbas')),
        ('TB', _('Tubérculos')),
        ('FL', _('Flores')),
        ('FRG', _('Forrajes')),
        ('OT', _('Otros')),
    ]

    field = models.ForeignKey(
        Field, on_delete=models.CASCADE, related_name="crops")
    name = models.CharField(max_length=100)
    crop_type = models.CharField(max_length=3, choices=CROP_TYPE_CHOICES)
    optimal_conditions = models.TextField()
    expected_harvest_date = models.DateField()
    num_plants = models.IntegerField(null=True, blank=True)
    yield_per_plant = models.FloatField(null=True, blank=True)
    market_price = models.FloatField(null=True, blank=True)

    # Dueño del cultivo
    farmer = models.ForeignKey(
        'farmer.Farmer', on_delete=models.CASCADE, related_name="crops")
    # Scout que registró el cultivo (opcional)
    scout = models.ForeignKey('scouts.Scout', on_delete=models.SET_NULL,
                              null=True, blank=True, related_name="crops_added")

    # Nuevos campos
    planting_date = models.DateField(null=True, blank=True)
    harvest_date = models.DateField(null=True, blank=True)
    yield_estimate = models.FloatField(null=True, blank=True)
    nutrient_requirements = models.JSONField(null=True, blank=True)

    def __str__(self):
        return self.name

# 🔹 MODELO CROP OBSERVATION (OBSERVACIONES)


class CropObservation(models.Model):
    crop = models.ForeignKey(Crop, on_delete=models.CASCADE,
                             related_name="observations", blank=True, null=True)
    date = models.DateField(auto_now_add=True)
    stand_count = models.IntegerField(
        help_text="Número de plantas en una unidad de medida definida")
    crop_height = models.FloatField(help_text="Altura del cultivo en cm")
    soil_temperature = models.FloatField(
        help_text="Temperatura del suelo en Celsius")
    weed_pressure = models.CharField(
        max_length=100, help_text="Descripción de la presión de malezas")
    weed_height = models.FloatField(help_text="Altura de las malezas en cm")
    pests = models.CharField(max_length=200, blank=True,
                             help_text="Descripción de las plagas observadas")
    rain_accumulation = models.FloatField(
        help_text="Acumulación de lluvia en mm")
    notes = models.TextField(blank=True, null=True,
                             help_text="Notas adicionales")

    # Indicar quién hizo la observación
    observer_scout = models.ForeignKey(
        'scouts.Scout', on_delete=models.SET_NULL, null=True, blank=True, related_name="observations")
    observer_farmer = models.ForeignKey(
        'farmer.Farmer', on_delete=models.SET_NULL, null=True, blank=True, related_name="observations_farmer")

    def save(self, *args, **kwargs):
        if self.observer_scout and self.observer_farmer:
            raise ValueError(
                "Una observación no puede ser registrada por ambos, Scout y Farmer.")
        super().save(*args, **kwargs)

    def __str__(self):
        return f"Observación de {self.crop.name} el {self.date.strftime('%Y-%m-%d')}"


class Brand(models.Model):
    PRODUCT_TYPES = [
        ('nitrogen', 'Nitrógeno'),
        ('phosphorus', 'Fósforo'),
        ('potassium', 'Potasio'),
    ]

    name = models.CharField(max_length=200)
    description = models.TextField(null=True, blank=True)
    website = models.URLField(null=True, blank=True)
    contact_info = models.CharField(max_length=200, null=True, blank=True)
    product_type = models.CharField(
        max_length=50, choices=PRODUCT_TYPES, null=True, blank=True)
    price_per_unit = models.FloatField(
        null=True, blank=True)  # Precio por unidad
    unit_of_measure = models.CharField(
        max_length=50, null=True, blank=True)  # Unidad (kg, litros, etc.)
    availability = models.BooleanField(
        default=True)  # Disponibilidad del producto

    def __str__(self):
        return self.name


class Notification(models.Model):
    scout = models.ForeignKey('scouts.Scout', on_delete=models.CASCADE,
                              related_name='notifications', null=True, blank=True)
    farmer = models.ForeignKey('farmer.Farmer', on_delete=models.CASCADE,
                               related_name='notifications', null=True, blank=True)  # 🔹 Corrección aquí

    message = models.TextField(null=True, blank=True)
    type = models.CharField(
        max_length=50,
        choices=[
            ('fertilization', 'Fertilization Alert'),
            ('soil_test', 'Soil Test Alert'),
            ('general', 'General Alert'),
        ],
        null=True,
        blank=True
    )
    status = models.CharField(
        max_length=10,
        choices=[
            ('unread', 'Unread'),
            ('read', 'Read'),
            ('urgent', 'Urgent'),
        ],
        default='unread'
    )
    notification_date = models.DateField(
        auto_now_add=True, null=True, blank=True)


class Recommendation(models.Model):
    """
    Modelo de recomendaciones tanto para cultivos (Crop) como para campos (Field).
    La descripción es generada por la IA y almacenada en `description`.
    """

    title = models.CharField(max_length=200)
    description = models.TextField(
        blank=True, null=True)  # ✅ Respuesta de la IA
    created_at = models.DateTimeField(auto_now_add=True, null=True, blank=True)

    # Recomendaciones para cultivos (Crop) o campos (Field)
    crop = models.ForeignKey('Crop', on_delete=models.CASCADE,
                             related_name='recommendations', null=True, blank=True)
    field = models.ForeignKey('Field', on_delete=models.CASCADE,
                              related_name='recommendations', null=True, blank=True)

    # Datos de origen de la recomendación
    crop_observation = models.ForeignKey(
        'CropObservation', on_delete=models.CASCADE, related_name='recommendations', null=True, blank=True)
    soil_test = models.ForeignKey(
        'SoilTest', on_delete=models.CASCADE, related_name='recommendations', null=True, blank=True)

    # 🔹 Relación con Brand para futuras recomendaciones de productos
    brand = models.ForeignKey(
        'Brand', on_delete=models.SET_NULL, null=True, blank=True)

    def save(self, *args, **kwargs):
        """
        Asegura que una recomendación solo se asocie a un `Field` o a un `Crop`, pero no a ambos.
        """
        if self.crop and self.field:
            raise ValueError(
                "Una recomendación no puede estar asociada a un campo (Field) y a un cultivo (Crop) al mismo tiempo.")
        if not self.crop and not self.field:
            raise ValueError(
                "Una recomendación debe estar asociada a un `Crop` o a un `Field`.")

        super().save(*args, **kwargs)

    def __str__(self):
        if self.crop:
            return f"Recomendación para cultivo {self.crop.name} ({self.title})"
        if self.field:
            return f"Recomendación para campo {self.field.name} ({self.title})"
        return self.title
