# forms.py

from django import forms
from django.shortcuts import render, redirect
from django.contrib.auth.models import User
from django.contrib.auth.forms import UserCreationForm
from .models import Scout, Field, Crop, CropObservation, SoilTest, SoilType
from farmer.models import Farmer

# forms.py


class UserRegistrationForm(forms.Form):
    USER_TYPE_CHOICES = [
        ('scout', 'Scout - Participa en la supervisión de cultivos y registra observaciones.'),
        ('farmer', 'Farmer - Administra los campos y cultivos, recibe reportes y recomendaciones.')
    ]

    user_type = forms.ChoiceField(
        choices=USER_TYPE_CHOICES,
        widget=forms.Select(attrs={'class': 'form-control'}),
        label="Tipo de Usuario"
    )


class ScoutRegistrationForm(forms.ModelForm):
    class Meta:
        model = Scout
        fields = ['first_name', 'last_name', 'email', 'phone', 'address']

    def save(self, commit=True):
        scout = super().save(commit=False)
        if commit:
            scout.save()
        return scout


class FarmerRegistrationForm(forms.ModelForm):
    class Meta:
        model = Farmer
        fields = ['first_name', 'last_name', 'email', 'phone', 'address']

    def save(self, commit=True):
        farmer = super().save(commit=False)
        if commit:
            farmer.save()
        return farmer


class FieldForm(forms.ModelForm):
    soil_type = forms.ModelChoiceField(
        queryset=SoilType.objects.all(),
        widget=forms.Select(attrs={'class': 'form-control'}),
        required=False
    )

    class Meta:
        model = Field
        fields = ['name', 'location', 'latitude',
                  'longitude', 'size', 'soil_type']
        widgets = {
            'name': forms.TextInput(attrs={'class': 'form-control'}),
            'location': forms.TextInput(attrs={'class': 'form-control'}),
            'latitude': forms.NumberInput(attrs={'class': 'form-control'}),
            'longitude': forms.NumberInput(attrs={'class': 'form-control'}),
            'size': forms.NumberInput(attrs={'class': 'form-control'}),
        }


class CropForm(forms.ModelForm):
    # Campos adicionales para los nutrientes
    nitrogen = forms.FloatField(label="Nitrógeno (N)", required=False)
    phosphorus = forms.FloatField(label="Fósforo (P)", required=False)
    potassium = forms.FloatField(label="Potasio (K)", required=False)
    num_plants = forms.IntegerField(
        widget=forms.NumberInput(attrs={'id': 'num_plants'})  # ID explícito
    )

    class Meta:
        model = Crop
        fields = [
            'name',
            'crop_type',
            'optimal_conditions',
            'expected_harvest_date',
            'field',
            'num_plants',  # Asegurar que está incluido
            'yield_per_plant',
            'market_price',
            'planting_date',
            'harvest_date',
            'yield_estimate'
        ]  # Excluimos 'nutrient_requirements' porque será calculado dinámicamente
        widgets = {
            'expected_harvest_date': forms.DateInput(attrs={'type': 'date'}),
            'planting_date': forms.DateInput(attrs={'type': 'date'}),
            'harvest_date': forms.DateInput(attrs={'type': 'date'}),
        }

    def save(self, commit=True):
        crop = super().save(commit=False)

        # Combinar los valores de nutrientes en un JSON
        crop.nutrient_requirements = {
            "N": self.cleaned_data.get('nitrogen', None),
            "P": self.cleaned_data.get('phosphorus', None),
            "K": self.cleaned_data.get('potassium', None)
        }

        if commit:
            crop.save()

        return crop


class CropObservationForm(forms.ModelForm):
    class Meta:
        model = CropObservation
        fields = [
            'stand_count',
            'crop_height',
            'soil_temperature',
            'weed_pressure',
            'weed_height',
            'pests',
            'rain_accumulation',
            'notes'
        ]


class ScoutProfileForm(forms.ModelForm):
    class Meta:
        model = Scout
        fields = ['first_name', 'last_name', 'email',
                  'phone', 'address', 'profile_picture']


class SoilTestForm(forms.ModelForm):
    class Meta:
        model = SoilTest
        fields = [
            'date',
            'pH',  # ✅ Se agrega el campo pH
            'nitrate_ppm',
            'phosphorus_ppm',
            'potassium_ppm',
            'zinc_ppm',
            'sulfur_ppm',
            'organic_matter_percentage',
            'bulk_density',
            'yield_target'
        ]
        widgets = {
            'date': forms.DateInput(attrs={'type': 'date'}),
        }
