# Django Core
from django.http import JsonResponse
from django.shortcuts import render, get_object_or_404, redirect
from django.http import JsonResponse, HttpResponse
from django.contrib import messages
from django.utils.translation import gettext as _
from django.utils.formats import sanitize_separators
from django.utils.timezone import now
from django.db import IntegrityError
from django.db.models import Count, Avg, Sum, F, FloatField, Q
from django.db.models.functions import ExtractYear

# Models
from scouts.models import Scout, Field, Crop, CropObservation, Recommendation, Notification, SoilTest
from .models import Machinery, Farmer, MachineryData, Expense

# Forms
from .forms import MachineryForm, ExpenseForm, MachineryDataForm, FarmerProfileForm
from scouts.forms import FieldForm, CropForm, CropObservationForm, SoilTestForm, SoilType

# Decorators
from agroapp.common.decorators import user_required

# External Libraries
from openpyxl import Workbook
from io import BytesIO
import json

# Python Standard Library
from decimal import Decimal
from datetime import date, datetime


from scouts.utils import generate_recommendations, generate_soil_recommendations

import random


from social_network.models import Comment, Post, Book, Video
from social_network.forms import PostForm, CommentForm
from django.contrib.contenttypes.models import ContentType

@user_required('farmer')
def farmer_dashboard(request):
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        raise ValueError(_("El ID del Finquero no se encuentra en la sesión"))

    # Obtener los scouts asociados al finquero
    scouts = Scout.objects.filter(farmer_id=farmer_id)
    total_scouts = scouts.count()

    # Obtener los campos gestionados por el finquero
    fields = Field.objects.filter(farmer_id=farmer_id)
    total_fields = fields.count()

    # Obtener los cultivos gestionados
    crops = Crop.objects.filter(field__in=fields)
    total_crops = crops.count()

    # Obtener todas las recomendaciones asociadas a los cultivos y campos del finquero
    recommendations = Recommendation.objects.filter(
        Q(crop__in=crops) | Q(field__in=fields)
        # 🔹 Ordenadas por las más recientes
    ).select_related('crop', 'field').order_by('-created_at')

    recommendations_count = recommendations.count()

    # Datos para los gráficos
    crops_by_type = crops.values('crop_type').annotate(total=Count('id'))
    observations_by_month = CropObservation.objects.filter(crop__in=crops).annotate(
        month=Count('date__month')).values('month').annotate(avg_height=Avg('crop_height'))

    # Datos para la maquinaria
    machineries = Machinery.objects.filter(farmer_id=farmer_id)
    total_machinery = machineries.count()

    # Si no hay datos
    no_data = total_scouts == 0 and total_fields == 0 and total_crops == 0 and total_machinery == 0

    context = {
        'scouts': scouts,
        'total_scouts': total_scouts,
        'total_fields': total_fields,
        'total_crops': total_crops,
        'crops_by_type': crops_by_type,
        'observations_by_month': observations_by_month,
        # ✅ Ahora enviamos las recomendaciones al template
        'recommendations': recommendations,
        'recommendations_count': recommendations_count,
        'total_machinery': total_machinery,
        'no_data': no_data,
    }

    return render(request, 'farmer/dashboard.html', context)


@user_required('farmer')
def machinery_fuel_consumption_data(request):
    """
    Genera los datos para el gráfico de consumo de combustible de la maquinaria usando datos reales.
    """
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        raise ValueError("El ID del Finquero no se encuentra en la sesión")

    # Obtener todas las maquinarias del finquero
    machineries = Machinery.objects.filter(farmer_id=farmer_id)

    # Definir los meses
    months = [
        _('Enero'), _('Febrero'), _('Marzo'), _(
            'Abril'), _('Mayo'), _('Junio'),
        _('Julio'), _('Agosto'), _('Septiembre'), _(
            'Octubre'), _('Noviembre'), _('Diciembre')
    ]

    # Año actual
    current_year = datetime.now().year

    # Obtener los datos de consumo de combustible por mes
    fuel_data = []
    available_months = set()  # Para almacenar los meses con datos

    for machinery in machineries:
        # Obtener el consumo de combustible real agrupado por mes
        monthly_consumption = (
            MachineryData.objects
            # Filtra solo por el año actual
            .filter(machinery=machinery, date__year=current_year)
            .values('date__month')  # Agrupar por mes
            # Sumar el consumo de combustible
            .annotate(total_fuel=Sum('fuel_consumption'))
            .order_by('date__month')
        )

        # Crear un diccionario con los valores predeterminados en 0 para cada mes
        consumption_by_month = {month: 0 for month in range(1, 13)}

        # Rellenar los valores con los datos obtenidos
        for entry in monthly_consumption:
            consumption_by_month[entry['date__month']] = entry['total_fuel']
            # Añadir los meses con datos
            available_months.add(entry['date__month'])

        # Convertir los datos a una lista ordenada de acuerdo a los meses
        fuel_data.append({
            "name": machinery.name,
            "data": [consumption_by_month[month] for month in range(1, 13)]
        })

    # Filtrar los meses disponibles
    months = [months[month - 1] for month in sorted(available_months)]

    return JsonResponse({"months": months, "fuel_data": fuel_data})


@user_required('farmer')
def search_scouts(request):
    """
    Vista para buscar Scouts por correo electrónico y asignarlos al finquero.
    """
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        raise ValueError(_("El ID del Finquero no se encuentra en la sesión"))

    farmer = get_object_or_404(Farmer, id=farmer_id)
    results = []
    message = None

    if request.method == 'GET' and 'email' in request.GET:
        email_query = request.GET.get('email', '').strip()
        results = Scout.objects.filter(email__icontains=email_query)

        if results.exists():
            for scout in results:
                if scout.farmer == farmer:
                    message = _("Este scout ya está asignado a ti.")
                elif scout.farmer is not None:
                    message = _("Este scout ya está asignado a otro finquero.")
        else:
            message = _(
                "No se encontraron scouts con este correo electrónico.")

    return render(request, 'farmer/search_scout.html', {
        'farmer': farmer,
        'results': results,
        'message': message,
    })


# ✅ GESTIÓN DE SCOUTS POR EL FARMER
@user_required('farmer')
def manage_scouts(request):
    farmer_id = request.session.get('farmer_id')
    scouts = Scout.objects.filter(farmer_id=farmer_id)
    return render(request, 'farmer/manage_scouts.html', {'scouts': scouts})


@user_required('farmer')
def assign_scout(request, scout_id):
    farmer_id = request.session.get('farmer_id')
    scout = get_object_or_404(Scout, id=scout_id)

    if request.method == 'POST':
        scout.farmer_id = farmer_id
        scout.save()
        messages.success(request, _("Scout asignado exitosamente."))
        return redirect('manage_scouts')


@user_required('farmer')
def unassign_scout(request, scout_id):
    farmer_id = request.session.get('farmer_id')
    scout = get_object_or_404(Scout, id=scout_id)

    if scout.farmer_id == farmer_id:
        scout.farmer_id = None
        scout.save()
        messages.success(request, _("Scout desasignado correctamente."))

    return redirect('manage_scouts')


# ✅ GESTIÓN DE CAMPOS POR EL FARMER
@user_required('farmer')
def farmer_field_list(request):
    farmer_id = request.session.get('farmer_id')
    fields = Field.objects.filter(farmer_id=farmer_id)
    has_crops = Crop.objects.filter(field__in=fields).exists()
    form = FieldForm()
    soil_type = SoilType.objects.all()
    return render(request, 'farmer/field_list.html', {'fields': fields, 'has_crops': has_crops, 'form': form, 'soil_type': soil_type})


@user_required('farmer')
def farmer_field_detail(request, field_id):
    farmer_id = request.session.get('farmer_id')
    field = get_object_or_404(Field, id=field_id, farmer_id=farmer_id)
    crops = field.crops.all()
    has_crops = crops.exists()

    return render(request, 'farmer/field_detail.html', {'field': field, 'crops': crops, 'has_crops': has_crops})


@user_required('farmer')
def create_field(request):
    if request.method == 'POST':
        form = FieldForm(request.POST)

        if form.is_valid():
            field = form.save(commit=False)
            field.farmer_id = request.session.get('farmer_id')
            field.save()
            messages.success(request, "Campo creado con éxito.")
            return redirect('farmer-field-list')

    else:
        form = FieldForm()

    return render(request, 'farmer/field_detail.html', {'form': form})


@user_required('farmer')
def edit_field(request, field_id):
    """Permite a un Farmer editar un campo y redirigir a la lista de campos."""
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        raise ValueError("El ID del Finquero no se encuentra en la sesión")

    field = get_object_or_404(Field, id=field_id, farmer_id=farmer_id)

    if request.method == 'POST':
        form = FieldForm(request.POST, instance=field)
        if form.is_valid():
            form.save()
            # ✅ Redirigir a la lista de campos
            return redirect('farmer-field-list')
    else:
        form = FieldForm(instance=field)

    return render(request, 'field_list.html', {'form': form, 'field': field})


@user_required('farmer')
def delete_field(request, field_id):
    """Permite a un Farmer eliminar un campo y redirigir a la lista de campos."""
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        raise ValueError("El ID del Finquero no se encuentra en la sesión")

    field = get_object_or_404(Field, id=field_id, farmer_id=farmer_id)

    if request.method == 'GET':
        field.delete()
        return redirect('farmer-field-list')  # ✅ Redirigir después de eliminar

    return render(request, 'field_list.html', {'field': field})


# ✅ GESTIÓN DE CULTIVOS POR EL FARMER
@user_required('farmer')
def farmer_crop_list(request):
    farmer_id = request.session.get('farmer_id')
    fields = Field.objects.filter(farmer_id=farmer_id)
    crops = Crop.objects.filter(field__in=fields)
    has_crops = crops.exists()
    has_fields = fields.exists()

    return render(request, 'farmer/crop_list.html', {'crops': crops, 'has_crops': has_crops, 'has_fields': has_fields, 'fields': fields})


@user_required('farmer')
def farmer_crop_detail(request, crop_id):
    """Muestra el detalle de un cultivo para un Farmer."""
    farmer_id = request.session.get('farmer_id')
    crop = get_object_or_404(Crop, id=crop_id, field__farmer_id=farmer_id)

    # Obtener el número de plantas
    if crop.observations.exists():
        last_observation = crop.observations.latest('date')
        num_plants = last_observation.stand_count
    else:
        num_plants = crop.num_plants

    observations = crop.observations.all()
    # ✅ Inicializar correctamente
    form = CropObservationForm(initial={'num_plants': num_plants})

    return render(request, 'farmer/crop_detail.html', {
        'crop': crop,
        'observations': observations,
        'form': form,
        'num_plants': num_plants
    })


@user_required('farmer')
def create_crop(request):
    """Permite a un Farmer registrar un cultivo en un campo y redirige a la página anterior después de guardarlo."""
    if request.method == 'POST':
        form = CropForm(request.POST)
        if form.is_valid():
            crop = form.save(commit=False)

            # ✅ Obtener el Farmer desde el campo relacionado
            farmer_id = request.session.get('farmer_id')
            if not farmer_id:
                messages.error(
                    request, "No se encontró un finquero en sesión.")
                return redirect('farmer-crop-list')

            # ✅ Asignar el Farmer al cultivo
            crop.farmer_id = farmer_id  # Asigna el ID directamente
            crop.save()

            messages.success(request, "Cultivo registrado con éxito.")

            # 🔹 Obtener la URL previa de donde se hizo la solicitud
            previous_url = request.META.get('HTTP_REFERER', 'farmer-crop-list')
            return redirect(previous_url)  # ✅ Volver a la página anterior

    else:
        form = CropForm()

        farmer_id = request.session.get('farmer_id')
        if not farmer_id:
            messages.error(request, "No se encontró un finquero en sesión.")
            return redirect('farmer-crop-list')

        fields = Field.objects.filter(farmer=farmer_id)

    return render(request, 'farmer/create_crop.html', {'form': form, 'fields': fields})


@user_required('farmer')
def edit_crop(request, crop_id):
    """Permite a un Farmer editar un cultivo y redirigir de vuelta al campo correspondiente."""
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        raise ValueError("El ID del Finquero no se encuentra en la sesión")

    crop = get_object_or_404(Crop, id=crop_id, farmer_id=farmer_id)

    # Obtener opciones de tipo de cultivo
    crop_type_choices = Crop.CROP_TYPE_CHOICES

    # Obtener todos los campos del Farmer
    fields = Field.objects.filter(farmer=farmer_id)

    # Obtener la URL de referencia (si existe)
    # ✅ Redirige al campo del cultivo si no hay `next`
    next_url = request.GET.get('next', f'/farmer/fields/{crop.field.id}/')

    if request.method == 'POST':
        form = CropForm(request.POST, instance=crop)
        if form.is_valid():
            crop = form.save(commit=False)

            # Procesar datos de nutrientes
            nitrogen = sanitize_separators(request.POST.get('nitrogen', '0'))
            phosphorus = sanitize_separators(
                request.POST.get('phosphorus', '0'))
            potassium = sanitize_separators(request.POST.get('potassium', '0'))

            crop.nutrient_requirements = {
                "N": float(nitrogen),
                "P": float(phosphorus),
                "K": float(potassium),
            }
            crop.save()

            return redirect(next_url)  # ✅ Redirige a la página de origen

    else:
        form = CropForm(instance=crop)

    return render(request, 'farmer/edit_crop.html', {
        'form': form,
        'crop': crop,
        'crop_type_choices': crop_type_choices,
        'fields': fields,
        'next_url': next_url,  # Pasar `next` a la plantilla
    })


@user_required('farmer')
def delete_crop(request, crop_id):
    """Permite a un Farmer eliminar un cultivo usando AJAX desde un modal."""

    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        print("❌ No se encontró el ID del Farmer en la sesión.")
        return JsonResponse({'success': False, 'message': "No se encontró el ID del Farmer en la sesión."}, status=403)

    crop = get_object_or_404(Crop, id=crop_id, farmer_id=farmer_id)

    # if request.method != 'POST':  # ✅ Solo permite `POST`
    #     print("❌ Método no permitido.")
    #     return JsonResponse({'success': False, 'message': "Método no permitido."}, status=405)

    print(f"✅ Cultivo {crop_id} eliminado correctamente.")

    crop.delete()

    return JsonResponse({'success': True, 'message': "Cultivo eliminado con éxito."}, status=200)


@user_required('farmer')
def create_observation(request, crop_id):
    """Permite a un Farmer registrar una observación en un cultivo y generar recomendaciones."""
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        return redirect('login')  # ✅ Redirigir a login si no está autenticado

    crop = get_object_or_404(Crop, id=crop_id, farmer_id=farmer_id)

    if request.method == 'POST':
        form = CropObservationForm(request.POST)
        if form.is_valid():
            observation = form.save(commit=False)
            observation.crop = crop
            # ✅ Registrar que la observación la hizo el Farmer
            observation.observer_farmer_id = farmer_id
            observation.save()

            # Obtener el análisis de suelo relacionado con el campo del cultivo
            soil_test = SoilTest.objects.filter(field=crop.field).first()

            if soil_test:
                try:
                    # ✅ Generar recomendaciones
                    generate_recommendations(soil_test, crop, observation)
                except Exception as e:
                    print(f"Error al generar la recomendación: {e}")

            # ✅ Redirigir al detalle del cultivo
            return redirect('farmer-crop-detail', crop_id=crop.id)

    return render(request, 'farmer/crop_detail.html', {'crop': crop, 'form': form})


@user_required('farmer')
def edit_observation(request, observation_id):
    """Permite a un Farmer editar una observación y redirigir automáticamente."""
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        return redirect('login')  # ✅ Redirigir a login si no está autenticado

    observation = get_object_or_404(
        CropObservation, id=observation_id, crop__farmer_id=farmer_id)

    if request.method == 'POST':
        form = CropObservationForm(request.POST, instance=observation)
        if form.is_valid():
            form.save()
            # ✅ Redirigir directamente
            return redirect('farmer-crop-detail', crop_id=observation.crop.id)

    return render(request, 'farmer/crop_detail.html', {'form': form, 'observation': observation})


@user_required('farmer')
def delete_observation(request, observation_id):
    """Permite a un Farmer eliminar una observación y redirigir automáticamente."""
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        return redirect('login')  # ✅ Redirigir a login si no está autenticado

    observation = get_object_or_404(
        CropObservation, id=observation_id, crop__farmer_id=farmer_id)
    crop_id = observation.crop.id

    if request.method == 'POST':
        observation.delete()
        # ✅ Redirigir directamente
        return redirect('farmer-crop-detail', crop_id=crop_id)

    return render(request, 'farmer/crop_detail.html', {'observation': observation})


@user_required('farmer')
def farmer_field_crop_types_data(request, field_id):
    field = get_object_or_404(Field, id=field_id)
    crops = field.crops.all()
    crop_types = crops.values('crop_type').annotate(count=Count('crop_type'))

    data = {
        'labels': [dict(Crop.CROP_TYPE_CHOICES).get(ctype['crop_type'], _('Desconocido')) for ctype in crop_types],
        'counts': [ctype['count'] for ctype in crop_types]
    }
    return JsonResponse(data)


@user_required('farmer')
def farmer_crop_type_distribution(request):
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        raise ValueError(_("El ID del Finquero no se encuentra en la sesión"))

    scouts = Scout.objects.filter(farmer_id=farmer_id)
    fields = Field.objects.filter(scout__in=scouts)
    crops = Crop.objects.filter(field__in=fields)

    crop_types_count = crops.values(
        'crop_type').annotate(count=Count('crop_type'))
    crop_types_labels = [dict(Crop.CROP_TYPE_CHOICES).get(
        ctype['crop_type'], _('Desconocido')) for ctype in crop_types_count]
    crop_types_data = [ctype['count'] for ctype in crop_types_count]

    return JsonResponse({
        'labels': crop_types_labels,
        'data': crop_types_data,
    })


@user_required('farmer')
def get_growth_trend_data(request, crop_id):
    crop = get_object_or_404(Crop, pk=crop_id)
    observations = CropObservation.objects.filter(crop=crop).order_by('date')

    data = {
        "dates": [obs.date.strftime("%Y-%m-%d") for obs in observations],
        "heights": [obs.crop_height for obs in observations],
    }

    return JsonResponse(data)


@user_required('farmer')
def get_soil_climate_data(request, field_id):
    field = get_object_or_404(Field, id=field_id)
    crops = field.crops.all()
    observations = CropObservation.objects.filter(crop__in=crops)

    if not observations.exists():
        return JsonResponse({'soil_conditions': [], 'heights': []})

    soil_conditions = observations.values(
        'date', 'soil_temperature', 'crop_height')
    data = {
        'dates': [obs['date'] for obs in soil_conditions],
        'soil_temperatures': [obs['soil_temperature'] for obs in soil_conditions],
        'crop_heights': [obs['crop_height'] for obs in soil_conditions]
    }
    return JsonResponse(data)


@user_required('farmer')
def farmer_recommendations(request):
    """Muestra todas las recomendaciones para los cultivos y campos de un Farmer, ordenadas por ID descendente."""
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        return redirect('login')

    # Obtener todos los cultivos del Farmer
    crops = Crop.objects.filter(field__farmer_id=farmer_id)

    # Obtener todos los campos del Farmer
    fields = Field.objects.filter(farmer_id=farmer_id)

    # Obtener recomendaciones ordenadas por ID descendente (mayor a menor)
    recommendations = Recommendation.objects.filter(
        Q(crop__in=crops) | Q(field__in=fields)
    ).select_related('crop_observation', 'soil_test').order_by('-id')  # ← Orden agregado aquí

    return render(request, 'farmer/recommendations.html', {
        'recommendations': recommendations
    })


@user_required('farmer')
def manage_machinery(request):
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        raise ValueError(_("El ID del Finquero no se encuentra en la sesión"))

    farmer = get_object_or_404(Farmer, id=farmer_id)
    machinery_list = Machinery.objects.filter(farmer=farmer)

    # Formulario para crear nueva maquinaria
    if request.method == 'POST':
        form = MachineryForm(request.POST)
        if form.is_valid():
            machinery = form.save(commit=False)
            machinery.farmer = farmer
            machinery.save()
            return redirect('manage-machinery')
    else:
        form = MachineryForm()

    scouts = Scout.objects.filter(farmer_id=farmer_id)

    # Pasar MACHINERY_TYPE_CHOICES al contexto
    machinery_type_choices = Machinery.MACHINERY_TYPE_CHOICES

    # 📌 Gráfico 1: Distribución de Tipos de Maquinaria
    type_counts = machinery_list.values(
        'machinery_type').annotate(count=Count('id'))
    machinery_types = [dict(Machinery.MACHINERY_TYPE_CHOICES)[
        # 🔹 Convertir clave a nombre legible
        t['machinery_type']] for t in type_counts]
    machinery_type_counts = [t['count'] for t in type_counts]

    # 📌 Gráfico 2: Edad Promedio de las Maquinarias por Tipo
    current_year = now().year
    machinery_age_groups = (
        machinery_list.values('machinery_type')
        .annotate(avg_age=Avg(current_year - ExtractYear('purchase_date')))
    )
    machinery_ages = [m['avg_age'] for m in machinery_age_groups]
    machinery_age_labels = [dict(Machinery.MACHINERY_TYPE_CHOICES)[
        m['machinery_type']] for m in machinery_age_groups]

    # 📌 Gráfico 3: Condición General de la Maquinaria
    condition_counts = machinery_list.values(
        'condition').annotate(count=Count('id'))
    machinery_conditions = [c['condition'] for c in condition_counts]
    machinery_conditions_counts = [c['count'] for c in condition_counts]

    return render(request, 'farmer/machinery_list.html', {
        'machinery_list': machinery_list,
        'form': form,
        'scouts': scouts,
        'machinery_type_choices': machinery_type_choices,
        'machinery_types': machinery_types,  # 🔹 Añadido para el gráfico 1
        'machinery_type_counts': machinery_type_counts,  # 🔹 Añadido para el gráfico 1
        'machinery_age_labels': machinery_age_labels,  # 🔹 Para el gráfico 2
        'machinery_ages': machinery_ages,  # 🔹 Para el gráfico 2
        'machinery_conditions': machinery_conditions,
        'machinery_conditions_counts': machinery_conditions_counts,
        # Lista correcta de Scouts
        'available_scouts': Scout.objects.filter(farmer=farmer),
    })


@user_required('farmer')
def edit_machinery(request, machinery_id):
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        raise ValueError(_("El ID del Finquero no se encuentra en la sesión"))

    farmer = get_object_or_404(Farmer, id=farmer_id)
    machinery = get_object_or_404(Machinery, id=machinery_id, farmer=farmer)

    if request.method == 'POST':
        form = MachineryForm(request.POST, instance=machinery)
        if form.is_valid():
            form.save()
            return redirect('manage-machinery')
    else:
        form = MachineryForm(instance=machinery)

    return render(request, 'farmer/machinery_edit.html', {
        'form': form,
        'machinery': machinery,
        # Lista correcta de Scouts
        'available_scouts': Scout.objects.filter(farmer=farmer),
    })


@user_required('farmer')
def delete_machinery(request, machinery_id):
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        raise ValueError(_("El ID del Finquero no se encuentra en la sesión"))

    farmer = get_object_or_404(Farmer, id=farmer_id)
    machinery = get_object_or_404(Machinery, id=machinery_id, farmer=farmer)

    if request.method == 'POST':
        machinery.delete()
        # Redirige a la lista de maquinaria después de eliminar
        return redirect('manage-machinery')

    return redirect('manage-machinery')  # En cualquier otro caso, redirige


@user_required('farmer')
def projected_gains_view(request):
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        raise ValueError(_("El ID del Finquero no se encuentra en la sesión"))

    crops = Crop.objects.filter(field__scout__farmer__id=farmer_id)
    projected_gains = []

    for crop in crops:
        num_plants = crop.num_plants or 0
        yield_per_plant = crop.yield_per_plant or 0
        market_price = crop.market_price or 0

        estimated_production = num_plants * yield_per_plant
        estimated_income = estimated_production * market_price

        projected_gains.append({
            'crop_name': crop.name,
            'num_plants': num_plants,
            'estimated_production': estimated_production,
            'market_price': market_price,
            'estimated_income': estimated_income,
        })

    return render(request, 'farmer/projected_gains.html', {
        'projected_gains': projected_gains
    })


@user_required('farmer')
def projected_gains_chart_data(request):
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        raise ValueError(_("El ID del Finquero no se encuentra en la sesión"))

    crops = Crop.objects.filter(field__scout__farmer__id=farmer_id)
    data = {
        'crop_names': [],
        'estimated_production': [],
        'estimated_income': []
    }

    for crop in crops:
        num_plants = crop.num_plants or 0
        yield_per_plant = crop.yield_per_plant or 0
        market_price = crop.market_price or 0

        estimated_production = num_plants * yield_per_plant
        estimated_income = estimated_production * market_price

        data['crop_names'].append(crop.name)
        data['estimated_production'].append(estimated_production)
        data['estimated_income'].append(estimated_income)

    return JsonResponse(data)


@user_required('farmer')
def report_list(request):
    """
    Muestra la lista de reportes disponibles para descargar.
    """
    reports = [
        {'title': _("Rendimiento de Cultivos"), 'description': _(
            "Resumen del rendimiento histórico y proyectado de los cultivos."), 'type': 'crop_performance'},
        {'title': _("Análisis de Maquinaria"), 'description': _(
            "Resumen del uso y estado actual de la maquinaria."), 'type': 'machinery_analysis'},
        {'title': _("Distribución de Campos y Cultivos"), 'description': _(
            "Análisis de los campos y los cultivos asignados."), 'type': 'field_distribution'},
    ]
    return render(request, 'farmer/report_list.html', {'reports': reports})


@user_required('farmer')
def generate_report(request, report_type):
    """
    Genera y descarga el reporte en formato Excel.
    """
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        raise ValueError(_("El ID del Finquero no se encuentra en la sesión"))

    # Crear el archivo Excel
    wb = Workbook()
    ws = wb.active
    ws.title = _("Reporte")

    # Crear datos según el tipo de reporte
    if report_type == 'crop_performance':
        ws.append([_("Nombre del Cultivo"), _(
            "Producción Estimada"), _("Ingresos Estimados")])
        crops = Crop.objects.filter(field__scout__farmer__id=farmer_id)
        for crop in crops:
            num_plants = crop.num_plants or 0
            yield_per_plant = crop.yield_per_plant or 0
            market_price = crop.market_price or 0
            estimated_production = num_plants * yield_per_plant
            estimated_income = estimated_production * market_price
            ws.append([crop.name, estimated_production, estimated_income])

    elif report_type == 'machinery_analysis':
        ws.append([_("Nombre"), _("Tipo"), _("Estado"), _("Asignado a")])
        machinery = Machinery.objects.filter(farmer_id=farmer_id)
        for item in machinery:
            ws.append([
                item.name,
                item.get_machinery_type_display(),
                item.condition,
                f"{item.assigned_to.first_name} {item.assigned_to.last_name}" if item.assigned_to else _(
                    "En Bodega")
            ])

    elif report_type == 'field_distribution':
        ws.append([_("Nombre del Campo"), _("Tamaño"), _("Tipo de Suelo")])
        fields = Field.objects.filter(scout__farmer__id=farmer_id)
        for field in fields:
            ws.append([field.name, field.size,
                      field.soil_type.name if field.soil_type else _("No definido")])

    # Serializar el archivo en memoria usando BytesIO
    output = BytesIO()
    wb.save(output)
    output.seek(0)

    # Preparar la respuesta HTTP para la descarga
    response = HttpResponse(
        content=output,
        content_type='application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
    )
    response['Content-Disposition'] = f'attachment; filename={report_type}.xlsx'
    return response


@user_required('farmer')
def field_area_comparison_farmer(request):
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        raise ValueError(_("El ID del Finquero no se encuentra en la sesión"))

    # Obtener todos los campos del finquero y sus scouts
    fields = Field.objects.filter(scout__farmer_id=farmer_id)

    # Extraer nombres y áreas
    names = [field.name for field in fields]
    areas = [field.size for field in fields]

    # Preparar datos para el gráfico
    data = {
        'names': names,
        'areas': areas,
    }

    return JsonResponse(data)


@user_required('farmer')
def field_climate_analysis_farmer(request):
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        raise ValueError(_("El ID del Finquero no se encuentra en la sesión"))

    # Obtener todos los campos del finquero
    fields = Field.objects.filter(scout__farmer_id=farmer_id)

    data = []
    for field in fields:
        data.append({
            'name': field.name,
            # Simulación de temperatura
            'temperature': round(15 + 10 * random.random(), 1),
            # Simulación de humedad
            'humidity': round(50 + 30 * random.random(), 1),
            # Simulación de precipitación
            'precipitation': round(0 + 10 * random.random(), 1)
        })

    return JsonResponse(data, safe=False)


@user_required('farmer')
def soil_type_distribution_farmer(request):
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        raise ValueError(_("El ID del Finquero no se encuentra en la sesión"))

    # Obtener la distribución de tipos de suelo en los campos del finquero
    soil_data = (
        Field.objects.filter(scout__farmer_id=farmer_id)
        .exclude(soil_type__isnull=True)
        # 🔹 Asegúrate de obtener el nombre, no el ID
        .values('soil_type__name')
        .annotate(total_area=Sum('size'))
    )

    # Formatear los datos para el gráfico
    soil_types = [soil['soil_type__name'] for soil in soil_data]
    soil_areas = [soil['total_area'] for soil in soil_data]

    # 👀 Verifica los datos en la consola
    print(f"Soil Data: {list(soil_data)}")

    return JsonResponse({'soil_types': soil_types, 'soil_areas': soil_areas})


@user_required('farmer')
def estimated_production_by_crop_type(request):
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        raise ValueError(_("El ID del Finquero no se encuentra en la sesión"))

    crops = Crop.objects.filter(field__scout__farmer__id=farmer_id)

    # Calcular la producción estimada por tipo de cultivo
    production_data = crops.values('crop_type').annotate(
        total_production=Sum(
            F('num_plants') * F('yield_per_plant'),
            output_field=FloatField()
        )
    )

    labels = [dict(Crop.CROP_TYPE_CHOICES).get(data['crop_type'],
                                               _('Desconocido')) for data in production_data]
    data = [data['total_production'] or 0 for data in production_data]

    return JsonResponse({'labels': labels, 'data': data})


@user_required('farmer')
def estimated_income_by_crop_type(request):
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        raise ValueError(_("El ID del Finquero no se encuentra en la sesión"))

    crops = Crop.objects.filter(field__scout__farmer__id=farmer_id)

    # Calcular los ingresos estimados por tipo de cultivo
    income_data = crops.values('crop_type').annotate(
        total_income=Sum(
            F('num_plants') * F('yield_per_plant') * F('market_price'),
            output_field=FloatField()
        )
    )

    labels = [dict(Crop.CROP_TYPE_CHOICES).get(
        data['crop_type'], _('Desconocido')) for data in income_data]
    data = [data['total_income'] or 0 for data in income_data]

    return JsonResponse({'labels': labels, 'data': data})


@user_required('farmer')
def soil_test_list(request, field_id):
    """Muestra el historial de análisis de suelo de un campo para Farmers."""
    field = get_object_or_404(
        Field, id=field_id, farmer=request.session.get('farmer_id'))
    soil_tests = SoilTest.objects.filter(field=field).order_by('-date')

    form = SoilTestForm()

    return render(request, 'farmer/soil_test_list.html', {
        'field': field,
        'soil_tests': soil_tests,
        'form': form,
    })


@user_required('farmer')
def add_soil_test(request, field_id):
    """Permite a un Farmer registrar un nuevo análisis de suelo desde un modal."""
    field = get_object_or_404(
        Field, id=field_id, farmer=request.session.get('farmer_id'))

    if request.method == 'POST':
        form = SoilTestForm(request.POST)
        if form.is_valid():
            soil_test = form.save(commit=False)
            soil_test.field = field
            soil_test.save()

            # Generar recomendaciones automáticas
            generate_soil_recommendations(soil_test)

            # ✅ Redirigir al listado de análisis de suelo después de guardar
            return redirect('farmer-soil-test-list', field_id=field.id)

    else:
        form = SoilTestForm()

    return render(request, 'farmer/soil_test_list.html', {'field': field, 'form': form})


@user_required('farmer')
def mark_all_read_farmer(request):
    """
    Permite que un Farmer marque todas sus notificaciones como leídas.
    """
    farmer_id = request.session.get('farmer_id')
    if farmer_id:
        Notification.objects.filter(
            farmer_id=farmer_id, status='unread').update(status='read')
        # Redirigir a la página anterior si está disponible
        previous_url = request.META.get('HTTP_REFERER', '/')
        return redirect(previous_url)

    return redirect('/')  # Redirige al inicio si no hay farmer_id


@user_required('farmer')
def manage_machinery_data(request):
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        raise ValueError(_("El ID del Finquero no se encuentra en la sesión"))

    farmer = get_object_or_404(Farmer, id=farmer_id)

    # Obtener datos de maquinaria para la vista
    machinery_data_list = MachineryData.objects.filter(
        machinery__farmer=farmer).exclude(id__isnull=True)
    machinery_list = Machinery.objects.filter(farmer=farmer)

    # 🔹 Convertir fechas a strings antes de enviarlas para JSON
    fuel_data = list(machinery_data_list.values(
        'date').annotate(total_fuel=Sum('fuel_consumption')))
    for item in fuel_data:
        item['date'] = item['date'].isoformat()  # Convertir fecha a string

    hours_data = list(machinery_data_list.values(
        'machinery__name').annotate(total_hours=Sum('operating_hours')))
    scatter_data = list(machinery_data_list.values(
        'fuel_consumption', 'operating_hours'))

    return render(
        request,
        "farmer/manage-data-machinery.html",
        {
            "machinery_data_list": machinery_data_list,
            "machinery_list": machinery_list,
            "fuel_data": json.dumps(fuel_data),
            "hours_data": json.dumps(hours_data),
            "scatter_data": json.dumps(scatter_data),
        }
    )


@user_required('farmer')
def create_machinery_data(request):
    if request.method == "POST":
        machinery_id = request.POST.get("machinery")
        fuel_consumption = request.POST.get("fuel_consumption")
        operating_hours = request.POST.get("operating_hours")
        date = request.POST.get("date")
        notes = request.POST.get("notes", "")

        if machinery_id and fuel_consumption and operating_hours and date:
            MachineryData.objects.create(
                machinery_id=machinery_id,
                fuel_consumption=fuel_consumption,
                operating_hours=operating_hours,
                date=date,
                notes=notes,
            )
            messages.success(
                request, "Datos de maquinaria creados correctamente")
        else:
            messages.error(
                request, "Todos los campos obligatorios deben completarse")

        # Ajusta la URL de redirección según tu sistema
        return redirect("manage_machinery_data")

    return redirect("manage_machinery_data")


@user_required('farmer')
def update_machinery_data(request, data_id):
    if request.method == "POST":
        print(
            f"Recibiendo actualización para ID {data_id} con datos: {request.POST}")

        try:
            machinery_data = MachineryData.objects.get(id=data_id)

            machinery_data.fuel_consumption = request.POST.get(
                "fuel_consumption")
            machinery_data.operating_hours = request.POST.get(
                "operating_hours")
            machinery_data.date = request.POST.get("date")
            machinery_data.notes = request.POST.get("notes", "")

            machinery_data.save()
            print("Datos actualizados correctamente ✅")

            messages.success(
                request, "Datos de maquinaria actualizados correctamente")
        except MachineryData.DoesNotExist:
            print("❌ Error: No se encontró el dato de maquinaria")
            messages.error(request, "No se encontró el dato de maquinaria")

        return redirect("manage_machinery_data")

    return redirect("manage_machinery_data")


def delete_machinery_data(request, pk):
    data = get_object_or_404(MachineryData, id=pk)
    data.delete()
    return redirect("manage_machinery_data")


@user_required('farmer')
def manage_expenses(request):
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        raise ValueError(_("El ID del Finquero no se encuentra en la sesión"))

    farmer = get_object_or_404(Farmer, id=farmer_id)
    expenses = Expense.objects.filter(farmer=farmer).order_by(
        "date")  # Ordenado por fecha ascendente

    # Agrupar gastos por tipo y fecha
    expense_by_type_over_time = list(
        expenses.values('date', 'expense_type')
        .annotate(total=Sum('amount'))
        .order_by('date')
    )

    # Formatear datos para ApexCharts
    expenses_chart_data = {}
    for item in expense_by_type_over_time:
        expense_type = item["expense_type"]
        date_str = item["date"].strftime('%Y-%m-%d')
        total = float(item["total"])

        if expense_type not in expenses_chart_data:
            expenses_chart_data[expense_type] = []

        expenses_chart_data[expense_type].append({"x": date_str, "y": total})

    # Convertir a JSON
    expenses_chart_json = json.dumps(expenses_chart_data)

    form = ExpenseForm()
    return render(request, "farmer/manage-expenses.html", {
        "expenses": expenses,
        "form": form,
        "expenses_chart_json": expenses_chart_json,
    })


@user_required('farmer')
def create_expense(request):
    if request.method == "POST":
        form = ExpenseForm(request.POST)
        if form.is_valid():
            expense = form.save(commit=False)
            expense.farmer = get_object_or_404(
                Farmer, id=request.session.get('farmer_id'))
            expense.save()
    return redirect("manage_expenses")


@user_required('farmer')
def edit_expense(request, pk):
    expense = get_object_or_404(Expense, id=pk)
    if request.method == "POST":
        form = ExpenseForm(request.POST, instance=expense)
        if form.is_valid():
            form.save()
    return redirect("manage_expenses")


@user_required('farmer')
def delete_expense(request, pk):
    expense = get_object_or_404(Expense, id=pk)
    if request.method == "POST":
        expense.delete()
    return redirect("manage_expenses")


@user_required('farmer')
def edit_farmer_profile(request, farmer_id):
    farmer = get_object_or_404(Farmer, id=farmer_id)

    if request.method == 'POST':
        form = FarmerProfileForm(request.POST, request.FILES, instance=farmer)
        if form.is_valid():
            form.save()
            messages.success(request, _('Perfil actualizado con éxito.'))
            return redirect('edit_farmer_profile', farmer_id=farmer.id)
    else:
        form = FarmerProfileForm(instance=farmer)

    return render(request, 'farmer/edit_farmer_profile.html', {'form': form})



#SOCIAL NETWORK

@user_required('farmer')
def post_list(request):
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        return redirect('login')  # Redirigir al login si no está autenticado

    # Obtener el ContentType para Farmer
    farmer_content_type = ContentType.objects.get_for_model(Farmer)

    # Obtener los posts del usuario actual
    posts = Post.objects.all().order_by('-created_at').prefetch_related('comments')

    # Pasar el ContentType y el ID del usuario a la plantilla
    return render(request, 'farmer/post_list.html', {
        'posts': posts,
        'farmer_content_type': farmer_content_type,
        'farmer_id': farmer_id,
    })

@user_required('farmer')
def create_post(request):
    farmer_id = request.session.get('farmer_id')
    farmer_content_type = ContentType.objects.get_for_model(Farmer)

    if request.method == 'POST':
        form = PostForm(request.POST)
        if form.is_valid():
            post = form.save(commit=False)
            post.author_type = farmer_content_type
            post.author_id = farmer_id
            post.save()  # Guardar el post en la base de datos
            return redirect('post_list')
    else:
        form = PostForm()

    posts = Post.objects.filter(author_type=farmer_content_type, author_id=farmer_id)
    return render(request, 'farmer/post_list.html', {'form': form, 'posts': posts})


@user_required('farmer')
def edit_post(request, post_id):
    farmer_id = request.session.get('farmer_id')
    farmer_content_type = ContentType.objects.get_for_model(Farmer)

    # Obtener el post solo si el autor es el usuario actual
    post = get_object_or_404(Post, id=post_id, author_type=farmer_content_type, author_id=farmer_id)

    if request.method == 'POST':
        form = PostForm(request.POST, instance=post)
        if form.is_valid():
            form.save()
            return redirect('post_list')  # Redirigir a la lista de posts
    else:
        form = PostForm(instance=post)

    # Mostrar el formulario de edición en la misma página de lista de posts
    posts = Post.objects.filter(author_type=farmer_content_type, author_id=farmer_id)
    return render(request, 'farmer/post_list.html', {'form': form, 'posts': posts, 'editing_post': post})    

@user_required('farmer')  # Asegúrate de que el usuario esté autenticado como Farmer
def delete_post(request, post_id):
    # Obtener el ID del usuario actual desde la sesión
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        messages.error(request, "No estás autenticado.")
        return redirect('login')  # Redirigir al login si no está autenticado

    # Obtener el ContentType para Farmer
    farmer_content_type = ContentType.objects.get_for_model(Farmer)

    # Obtener el post solo si el autor es el usuario actual
    post = get_object_or_404(Post, id=post_id, author_type=farmer_content_type, author_id=farmer_id)

    if request.method == 'POST':
        # Eliminar el post
        post.delete()
        messages.success(request, "La publicación se ha eliminado correctamente.")
        return redirect('post_list')  # Redirigir a la lista de posts

    # Si no es una solicitud POST, redirigir a la lista de posts
    return redirect('post_list')

@user_required('farmer')  # Asegúrate de que el usuario esté autenticado como Farmer
def add_comment(request, post_id):
    # Obtener el ID del usuario actual desde la sesión
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        messages.error(request, "No estás autenticado.")
        return redirect('login')  # Redirigir al login si no está autenticado

    # Obtener el ContentType para Farmer
    farmer_content_type = ContentType.objects.get_for_model(Farmer)

    # Obtener el post al que se agregará el comentario
    post = get_object_or_404(Post, id=post_id)

    if request.method == 'POST':
        form = CommentForm(request.POST)
        if form.is_valid():
            comment = form.save(commit=False)
            comment.post = post
            comment.author_type = farmer_content_type
            comment.author_id = farmer_id
            comment.save()
            messages.success(request, "El comentario se ha agregado correctamente.")
            return redirect('post_list')  # Redirigir a la lista de posts
    else:
        form = CommentForm()

    # Si no es una solicitud POST, redirigir a la lista de posts
    return redirect('post_list')


@user_required('farmer')  # Asegúrate de que el usuario esté autenticado como Farmer
def edit_comment(request, comment_id):
    # Obtener el ID del usuario actual desde la sesión
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        messages.error(request, "No estás autenticado.")
        return redirect('login')  # Redirigir al login si no está autenticado

    # Obtener el ContentType para Farmer
    farmer_content_type = ContentType.objects.get_for_model(Farmer)

    # Obtener el comentario solo si el autor es el usuario actual
    comment = get_object_or_404(Comment, id=comment_id, author_type=farmer_content_type, author_id=farmer_id)

    if request.method == 'POST':
        form = CommentForm(request.POST, instance=comment)
        if form.is_valid():
            form.save()
            messages.success(request, "El comentario se ha editado correctamente.")
            return redirect('post_list')  # Redirigir a la lista de posts
    else:
        form = CommentForm(instance=comment)

    # Pasar el formulario a la plantilla
    posts = Post.objects.filter(author_type=farmer_content_type, author_id=farmer_id)
    return render(request, 'farmer/post_list.html', {
        'form': form,
        'posts': posts,
        'editing_comment': comment,  # Para identificar que estamos editando un comentario
    })


@user_required('farmer')  # Asegúrate de que el usuario esté autenticado como Farmer
def delete_comment(request, comment_id):
    # Obtener el ID del usuario actual desde la sesión
    farmer_id = request.session.get('farmer_id')
    if not farmer_id:
        messages.error(request, "No estás autenticado.")
        return redirect('login')  # Redirigir al login si no está autenticado

    # Obtener el ContentType para Farmer
    farmer_content_type = ContentType.objects.get_for_model(Farmer)

    # Obtener el comentario solo si el autor es el usuario actual
    comment = get_object_or_404(Comment, id=comment_id, author_type=farmer_content_type, author_id=farmer_id)

    if request.method == 'POST':
        comment.delete()
        messages.success(request, "El comentario se ha eliminado correctamente.")
        return redirect('post_list')  # Redirigir a la lista de posts

    # Si no es una solicitud POST, redirigir a la lista de posts
    return redirect('post_list')

            

def recursos_farmer(request):
    books = Book.objects.all().order_by('-uploaded_at')
    videos = Video.objects.all().order_by('-uploaded_at')
    return render(request, 'farmer/recursos_farmer.html', {
        'books': books,
        'videos': videos
    })            



def agriculture_course_view(request):
    return render(request, 'farmer/curso-agro.html', {
        'title': 'Curso de Agricultura de Precisión y Sostenible'
    })