from django.db import models
from django.utils.translation import gettext_lazy as _
# from scouts.models import Scout  # Asegúrate de que el modelo Scout esté importado correctamente


class Farmer(models.Model):
    first_name = models.CharField(max_length=100)
    last_name = models.CharField(max_length=100)
    email = models.EmailField(unique=True)
    phone = models.CharField(max_length=15, blank=True, null=True)
    address = models.CharField(max_length=255, blank=True, null=True)
    profile_picture = models.ImageField(
        upload_to='profile_pictures/', blank=True, null=True)

    def __str__(self):
        return f"{self.first_name} {self.last_name}"


class Machinery(models.Model):
    MACHINERY_TYPE_CHOICES = [
        ('tractor', _('Tractor')),
        ('combine', _('Cosechadora')),
        ('plow', _('Arado')),
        ('sprayer', _('Pulverizador')),
        ('seeder', _('Sembradora')),
        ('irrigation', _('Sistema de Riego')),
        ('other', _('Otro')),
    ]

    name = models.CharField(max_length=100, verbose_name=_('Nombre'))
    machinery_type = models.CharField(
        max_length=50, choices=MACHINERY_TYPE_CHOICES, verbose_name=_('Tipo de Maquinaria'))
    purchase_date = models.DateField(verbose_name=_('Fecha de Compra'))
    condition = models.CharField(max_length=50, verbose_name=_('Condición'))
    description = models.TextField(
        blank=True, null=True, verbose_name=_('Descripción'))
    farmer = models.ForeignKey(
        'farmer.Farmer',
        on_delete=models.CASCADE,
        related_name='machineries',
        verbose_name=_('Finquero'),
        help_text=_('El propietario de la maquinaria.')
    )

    assigned_to = models.ForeignKey(
        'scouts.Scout',
        on_delete=models.SET_NULL,
        null=True,
        blank=True,
        related_name='assigned_machinery',
        verbose_name=_('Asignado a'),
        help_text=_(
            'Indica quién tiene asignada la maquinaria. Déjelo en blanco si está en bodega.')
    )

    def __str__(self):
        return f"{self.name} - {self.get_machinery_type_display()}"


class MachineryData(models.Model):
    machinery = models.ForeignKey(Machinery, on_delete=models.CASCADE)
    fuel_consumption = models.FloatField()
    operating_hours = models.FloatField()
    date = models.DateField()
    notes = models.TextField(blank=True, null=True)
    # 🔹 Agregar si no está definido
    created_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return f"{self.machinery} - {self.date}"


class Expense(models.Model):
    EXPENSE_TYPE_CHOICES = [
        ('fertilizer', 'Fertilizante'),
        ('pesticide', 'Pesticida'),
        ('repair', 'Reparación de Equipos'),
        ('labor', 'Mano de Obra'),
        ('other', 'Otro'),
    ]

    farmer = models.ForeignKey(
        Farmer, on_delete=models.CASCADE, related_name='expenses')
    expense_type = models.CharField(
        max_length=50, choices=EXPENSE_TYPE_CHOICES)
    description = models.CharField(max_length=255)
    amount = models.DecimalField(max_digits=10, decimal_places=2)
    date = models.DateField()
    notes = models.TextField(blank=True, null=True)

    def __str__(self):
        return f"{self.get_expense_type_display()} - {self.description} (${self.amount})"
