# Librerías estándar
from datetime import timedelta

# Librerías de terceros
from django.contrib import messages
from django.contrib.auth.decorators import login_required
from django.db.models import Avg, Count, Q
from django.db.models.functions import TruncMonth
from django.http import JsonResponse
from django.shortcuts import get_object_or_404, redirect, render
from django.template.loader import render_to_string
from django.utils import timezone
from django.views.decorators.csrf import csrf_exempt
from django.utils.timezone import now, timedelta
from django.contrib.auth.decorators import login_required
from django.views.decorators.http import require_POST
from django.forms import modelform_factory

# Librerías locales
from scouts.models import Crop, CropObservation, Field, Scout, Recommendation, Brand
from scouts.views import get_current_scout
from social.models import News
from social_network.models import Comment, Post
from farmer.models import Farmer
from .forms import NewsForm, ScoutForm, BrandForm
from social_network.forms import PostForm
from agroapp.common.decorators import user_required

from agronomist.models import Agronomist
from agronomist.forms import AgronomistProfileForm

from django.http import HttpResponse
from openpyxl import Workbook
from django.utils.translation import gettext as _

from social_network.models import Comment, Post, Book, Video
from social_network.forms import PostForm, CommentForm
from django.contrib.contenttypes.models import ContentType

import json

# DASHBOARD

@user_required('agronomist')
def agronomist_dashboard(request):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

    # Obtener los scouts asignados al agrónomo
    scouts = Scout.objects.all()

    scout_data = []
    for scout in scouts:
        total_fields = Field.objects.filter(scout=scout).count()
        total_crops = Crop.objects.filter(field__scout=scout).count()
        recent_observations = CropObservation.objects.filter(
            crop__field__scout=scout,
            date__gte=timezone.now() - timedelta(days=30)
        ).count()
        critical_crops = Crop.objects.filter(
            field__scout=scout,
            # Cultivos con fechas de cosecha vencidas
            expected_harvest_date__lt=timezone.now()
        ).count()

        # Determinar el estado del scout
        scout_status = "Bueno"
        if critical_crops > 5:
            scout_status = "Crítico"
        elif recent_observations < 3:
            scout_status = "Moderado"

        scout_data.append({
            "id": scout.id,
            "name": f"{scout.first_name} {scout.last_name}",
            "total_fields": total_fields,
            "total_crops": total_crops,
            "recent_observations": recent_observations,
            "critical_crops": critical_crops,
            "status": scout_status,
        })

    return render(request, 'admin/admin_dashboard.html', {
        'scout_data': scout_data,
    })


@user_required('agronomist')
def scout_performance_data(request):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

    # Obtener todos los Scouts
    scouts = Scout.objects.all()

    # Si no hay scouts, devolver un error manejado o datos vacíos
    if not scouts.exists():
        return JsonResponse({
            'error': _('No hay scouts disponibles para mostrar datos de rendimiento.'),
            'labels': [],
            'data': []
        }, status=404)

    # Crear etiquetas y datos para el gráfico
    labels = [f"{scout.first_name} {scout.last_name}" for scout in scouts]
    # Usando el related_name correcto
    data = [scout.fields_managed.count() for scout in scouts]

    return JsonResponse({
        'labels': labels,
        'data': data
    })


# SCOUTS

@user_required('agronomist')
def scout_list(request):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))
    """
    Vista para listar a los Scouts con sus datos y el Finquero relacionado.
    """
    scouts = Scout.objects.select_related('farmer').all(
    )  # Optimizar con `select_related` si hay relación directa con Farmer
    return render(request, 'admin/scout_list.html', {'scouts': scouts})


@user_required('agronomist')
def farmer_details(request, farmer_id):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))
    """
    Retorna los detalles de un Finquero en formato JSON para el modal.
    """
    farmer = get_object_or_404(Farmer, id=farmer_id)
    data = {
        'name': f"{farmer.first_name} {farmer.last_name}",
        'email': farmer.email,
        'phone': farmer.phone,
        'address': farmer.address
    }
    return JsonResponse(data)


# RECOMENDACIONES

@user_required('agronomist')
def recommendations_dashboard(request):
    """
    Renderiza el dashboard de recomendaciones para el agrónomo.
    """
    recommendations = Recommendation.objects.select_related(
        "crop", "field").all()

    context = {
        "recommendations": recommendations
    }

    return render(request, "admin/recommendations_dashboard.html", context)


def recommendations_dashboard_data(request):
    """
    API que devuelve datos clave para los gráficos del dashboard.
    """
    total_crops = Crop.objects.count()

    # Definir cultivos en alerta: se considerarán aquellos con observaciones críticas
    alert_crops = CropObservation.objects.filter(
        crop_height__lt=20,  # Cultivos con altura baja
        weed_pressure__icontains="alta",  # Malezas altas
        pests__isnull=False  # Plagas detectadas
    ).values_list("crop_id", flat=True).distinct().count()

    # Rendimiento promedio estimado
    average_yield = Crop.objects.exclude(yield_estimate__isnull=True).aggregate(
        avg_yield=Avg("yield_estimate"))["avg_yield"] or 0

    data = {
        "total_crops": total_crops,
        "alert_crops": alert_crops,
        "average_yield": round(average_yield, 0),
    }
    return JsonResponse(data)


def get_recommendation(request, recommendation_id):
    """ Retorna los datos de una recomendación en formato JSON. """
    recommendation = get_object_or_404(Recommendation, id=recommendation_id)
    data = {
        "id": recommendation.id,
        "title": recommendation.title,
        "description": recommendation.description,
    }
    return JsonResponse(data)


def update_recommendation(request, recommendation_id):
    recommendation = get_object_or_404(Recommendation, id=recommendation_id)

    if request.method == "POST":
        title = request.POST.get("title", "").strip()
        description = request.POST.get("description", "").strip()

        if not title or not description:
            messages.error(request, "Todos los campos son obligatorios.")
            # No actualiza si hay error
            return redirect("recommendations-dashboard")

        recommendation.title = title
        recommendation.description = description
        recommendation.save()

        messages.success(request, "Recomendación actualizada correctamente.")
        # Reemplázalo con tu vista de éxito
        return redirect("recommendations-dashboard")

    messages.error(request, "Método no permitido.")
    return redirect("recommendations-dashboard")


def delete_recommendation(request, recommendation_id):
    # Obtener la recomendación o devolver un 404 si no existe
    recommendation = get_object_or_404(Recommendation, id=recommendation_id)

    if request.method == "POST":  # Asegurarse de que la solicitud sea POST
        recommendation.delete()  # Eliminar la recomendación
        messages.success(request, "Recomendación eliminada correctamente.")
    else:
        messages.error(request, "Método no permitido.")

    return redirect("recommendations-dashboard")  # Redirigir al dashboard


@user_required('agronomist')
def recent_observations_data(request):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

    scouts = Scout.objects.all()  # Ajusta esta consulta según tu modelo

    labels = [f"{scout.first_name} {scout.last_name}" for scout in scouts]
    recent_observations = [
        CropObservation.objects.filter(
            crop__field__scout=scout,
            date__gte=timezone.now() - timedelta(days=30)
        ).count()
        for scout in scouts
    ]

    data = {
        "labels": labels,
        "recent_observations": recent_observations,
    }
    return JsonResponse(data)


# FUNCIONES PARA MARCAS

# Lista de marcas
def brand_list(request):
    brands = Brand.objects.all()
    form = BrandForm()
    product_types = Brand.PRODUCT_TYPES  # Obtener los tipos de productos del modelo

    return render(request, "admin/brand_list.html", {
        "brands": brands,
        "form": form,
        "product_types": product_types,  # Pasar los tipos de productos al template
    })

# Crear una marca


def brand_create(request):
    if request.method == "POST":
        form = BrandForm(request.POST)
        if form.is_valid():
            form.save()
            messages.success(request, _("Marca creada correctamente."))
    return redirect("brand-list")

# Editar una marca


def brand_update(request, brand_id):
    brand = get_object_or_404(Brand, id=brand_id)
    if request.method == "POST":
        form = BrandForm(request.POST, instance=brand)
        if form.is_valid():
            form.save()
            messages.success(request, _("Marca actualizada correctamente."))
    return redirect("brand-list")

# Eliminar una marca


def brand_delete(request, brand_id):
    brand = get_object_or_404(Brand, id=brand_id)
    if request.method == "POST":
        brand.delete()
        messages.success(request, _("Marca eliminada correctamente."))
    return redirect("brand-list")

# @user_required('agronomist')
# def recommendations_crop_data(request):
#     """Retorna datos de cultivos para la tabla de recomendaciones."""
#     crops = Crop.objects.select_related('field').all()

#     data = []
#     for crop in crops:
#         data.append({
#             "id": crop.id,
#             "name": crop.name,
#             "field": crop.field.name if crop.field else _("Sin campo asignado"),
#             "type": crop.crop_type,
#             "expected_yield": f"{crop.expected_yield} kg/ha",
#             "last_observation": crop.last_observation.strftime("%Y-%m-%d") if crop.last_observation else _("Sin observación"),
#             "status": crop.get_status_display(),  # Asumiendo que `status` es un `choices`
#         })

#     return JsonResponse({"data": data})

# @user_required('agronomist')
# def recommendations_dashboard_data(request):
#     """Retorna métricas generales de cultivos para el dashboard."""
#     total_crops = Crop.objects.count()
#     alert_crops = Crop.objects.filter(status='alerta').count()  # Ajusta según el estado de alerta
#     average_yield = Crop.objects.aggregate(avg_yield=Avg('expected_yield'))['avg_yield'] or 0

#     return JsonResponse({
#         "total_crops": total_crops,
#         "alert_crops": alert_crops,
#         "average_yield": f"{average_yield:.2f} kg/ha"
#     })


@user_required('agronomist')
def detected_problems_data(request):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

    # Obtener todos los Scouts
    scouts = Scout.objects.all()

    # Verificar si hay scouts
    if not scouts.exists():
        return JsonResponse({
            'error': _('No hay scouts disponibles para mostrar datos de problemas detectados.'),
            'problems': []
        }, status=404)

    # Crear datos de problemas detectados
    problems = []
    for scout in scouts:
        pests = CropObservation.objects.filter(
            crop__field__scout=scout,
            pests__isnull=False
        ).count()
        deficiencies = CropObservation.objects.filter(
            crop__field__scout=scout,
            notes__icontains="deficiencia"
        ).count()
        others = CropObservation.objects.filter(
            crop__field__scout=scout,
            notes__icontains="problema"
        ).count()
        problems.append({
            "scout": f"{scout.first_name} {scout.last_name}",
            "pests": pests,
            "deficiencies": deficiencies,
            "others": others
        })

    return JsonResponse({'problems': problems})


# FUNCIONES DE NOTICIAS

@user_required('agronomist')
def news_list(request):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

    news = News.objects.all().order_by('-created_at')
    return render(request, 'admin/news_list.html', {'news': news, 'author': agronomist_id})


@user_required('agronomist')
def news_create(request):

    if request.method == 'POST':
        form = NewsForm(request.POST)
        if form.is_valid():
            news = form.save(commit=False)
            # news.author = author  # Asignar el autor como el scout actual
            news.save()
            return redirect('news-list')
    else:
        form = NewsForm()

    return render(request, 'news/admin/news_form.html', {'form': form})


@user_required('agronomist')
def news_update(request, pk):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

    news = get_object_or_404(News, pk=pk)

    if request.method == 'POST':
        news.title = request.POST.get('title')
        news.content = request.POST.get('content')
        news.save()
        messages.success(request, 'Noticia actualizada exitosamente.')
        return redirect('news-list')

    return render(request, 'admin/news_list.html', {'news': news})


@user_required('agronomist')
def news_delete(request, pk):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

    news = get_object_or_404(News, pk=pk)
    if request.method == 'POST':
        news.delete()
        messages.success(request, 'Noticia eliminada exitosamente.')
        return redirect('news-list')

    return render(request, 'admin/news_list.html', {'news': news})


# FUNCIONES SCOUTS

@user_required('agronomist')
def scout_list(request):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

    scouts = Scout.objects.all()
    return render(request, 'admin/scout_list.html', {'scouts': scouts})


@user_required('agronomist')
def scout_create(request):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

    if request.method == 'POST':
        form = ScoutForm(request.POST, request.FILES)
        if form.is_valid():
            form.save()
            messages.success(request, "Scout creado con éxito.")
            return redirect('scout-list')
    else:
        form = ScoutForm()
    return render(request, 'admin/scout_list.html', {'form': form})


@user_required('agronomist')
def scout_update(request, pk):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

    scout = get_object_or_404(Scout, pk=pk)
    if request.method == 'POST':
        form = ScoutForm(request.POST, request.FILES, instance=scout)
        if form.is_valid():
            form.save()
            messages.success(request, "Scout actualizado con éxito.")
            return redirect('scout-list')
    else:
        form = ScoutForm(instance=scout)
    return render(request, 'admin/scout_list.html', {'form': form})


@user_required('agronomist')
def scout_delete(request, pk):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

    scout = get_object_or_404(Scout, pk=pk)
    if request.method == 'POST':
        scout.delete()
        messages.success(request, "Scout eliminado con éxito.")
        return redirect('scout-list')
    return render(request, 'admin/scout_list.html', {'scout': scout})


# FUNCIONES PARA PUBLICACIONES

# Listar publicaciones
@user_required('agronomist')
def post_list(request):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

    posts = Post.objects.prefetch_related(
        'comments').all()  # Optimiza la carga de comentarios
    return render(request, 'admin/admin_post_list.html', {'posts': posts})

# Crear publicación


@user_required('agronomist')
def post_create(request):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

    if request.method == 'POST':
        form = PostForm(request.POST)
        if form.is_valid():
            post = form.save(commit=False)
            post.author = request.user.scout_profile  # Asignar el autor actual
            post.save()
            return JsonResponse({'success': True})
    return JsonResponse({'success': False}, status=400)

# Actualizar publicación


@user_required('agronomist')
def post_update(request, post_id):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

    post = get_object_or_404(Post, id=post_id)
    # Crear formulario dinámicamente
    PostForm = modelform_factory(Post, fields=['title', 'content'])

    if request.method == 'POST':
        form = PostForm(request.POST, instance=post)
        if form.is_valid():
            form.save()
            # Redirigir a la página de gestión de publicaciones
            return redirect('post-list')

    return JsonResponse({'success': False}, status=400)


# Eliminar publicación
@user_required('agronomist')
def post_delete(request, post_id):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

    post = get_object_or_404(Post, id=post_id)

    if request.method == 'POST':
        post.delete()
        messages.success(request, "Publicación eliminada correctamente.")
        # Reemplázalo con la vista a donde quieres redirigir
        return redirect("post-list")

    messages.error(request, "Método no permitido.")
    return redirect("post-list")


def load_comments(request, post_id):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

    if request.method == 'GET':
        post = get_object_or_404(Post, id=post_id)
        comments_html = render_to_string(
            'admin/comment_list.html', {'comments': post.comments.all()})
        return JsonResponse({'html': comments_html})

# Crear comentario


@csrf_exempt
def comment_create(request, post_id):
    post = get_object_or_404(Post, id=post_id)
    if request.method == 'POST':
        content = request.POST.get('content')
        if content:
            comment = Comment.objects.create(
                post=post, author=request.user.scout_profile, content=content)
            return JsonResponse({
                'success': True,
                'html': render_to_string('admin/admin_post_list.html', {'comment': comment}),
            })
    return JsonResponse({'success': False}, status=400)

# Actualizar comentario


@csrf_exempt
def comment_update(request, comment_id):
    comment = get_object_or_404(Comment, id=comment_id)
    if request.method == 'POST':
        content = request.POST.get('content')
        if content:
            comment.content = content
            comment.save()
            return JsonResponse({'success': True})
    return JsonResponse({'success': False}, status=400)

# Eliminar comentario


@csrf_exempt
def comment_delete(request, comment_id):
    comment = get_object_or_404(Comment, id=comment_id)
    if request.method == 'POST':
        comment.delete()
        return JsonResponse({'success': True})
    return JsonResponse({'success': False}, status=400)


# @user_required('agronomist')
# def admin_recommendation_dashboard(request):
#     agronomist_id = request.session.get('agronomist_id')
#     if not agronomist_id:
#         raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

#     """Renderiza el panel de recomendaciones."""
#     return render(request, 'admin/recommendation_dashboard.html')

# @user_required('agronomist')
# def get_problem_trends(request):
#     agronomist_id = request.session.get('agronomist_id')
#     if not agronomist_id:
#         raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

#     """Devuelve las tendencias de problemas en cultivos por mes."""
#     trends = (
#         CropObservation.objects
#         .filter(date__gte=now() - timedelta(days=180))  # Últimos 6 meses
#         .annotate(month=TruncMonth('date'))
#         .values('month')
#         .annotate(
#             weed_issues=Count('id', filter=Q(weed_pressure__icontains='alta')),
#             pest_issues=Count('id', filter=Q(pests__isnull=False)),
#             rain_deficit=Count('id', filter=Q(rain_accumulation__lt=50)),  # Acumulación crítica
#         )
#         .order_by('month')
#     )
#     return JsonResponse({'data': list(trends)})

# @user_required('agronomist')
# def get_recommendation_summary(request):
#     agronomist_id = request.session.get('agronomist_id')
#     if not agronomist_id:
#         raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))


#     """Devuelve datos para el gráfico de recomendaciones pendientes y completadas."""
#     recommendation_data = (
#         Recommendation.objects
#         .values('title')
#         .annotate(
#             pending=Count('id', filter=Q(status='pending')),
#             completed=Count('id', filter=Q(status='completed')),
#         )
#     )
#     return JsonResponse({'data': list(recommendation_data)})

# @user_required('agronomist')
# def select_crop_for_recommendation(request):
#     agronomist_id = request.session.get('agronomist_id')
#     if not agronomist_id:
#         raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

#     """Vista para listar y filtrar cultivos disponibles para recomendaciones."""
#     crops = Crop.objects.all()

#     # Filtrar cultivos según parámetros del formulario (opcional)
#     if 'field_name' in request.GET:
#         crops = crops.filter(field__name__icontains=request.GET['field_name'])
#     if 'scout' in request.GET:
#         crops = crops.filter(field__scout__first_name__icontains=request.GET['scout'])
#     if 'crop_type' in request.GET:
#         crops = crops.filter(crop_type=request.GET['crop_type'])

#     return render(request, 'admin/select_crop.html', {'crops': crops})

# @user_required('agronomist')
# def create_recommendation(request, crop_id):
#     agronomist_id = request.session.get('agronomist_id')
#     if not agronomist_id:
#         raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

#     """Vista para registrar una recomendación para un cultivo específico."""
#     crop = get_object_or_404(Crop, id=crop_id)
#     if request.method == 'POST':
#         title = request.POST.get('title')
#         description = request.POST.get('description')
#         Recommendation.objects.create(title=title, description=description, crop=crop)
#         messages.success(request, f"Recomendación creada para el cultivo {crop.name}.")
#         return redirect('admin-recommendation')

#     return render(request, 'admin/create_recommendation.html', {'crop': crop})

# @user_required('agronomist')
# def recommendation_list(request):
#     agronomist_id = request.session.get('agronomist_id')
#     if not agronomist_id:
#         raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

#     """Vista principal para listar las recomendaciones."""
#     crops = Crop.objects.all()
#     recommendations = Recommendation.objects.all()
#     return render(request, 'admin/recommendation_list.html', {'crops': crops, 'recommendations': recommendations})

# @csrf_exempt
# def recommendation_create(request):

#     """Crear una nueva recomendación."""
#     if request.method == 'POST':
#         crop_id = request.POST.get('crop_id')
#         title = request.POST.get('title')
#         description = request.POST.get('description')
#         crop = get_object_or_404(Crop, id=crop_id)
#         recommendation = Recommendation.objects.create(crop=crop, title=title, description=description)
#         return JsonResponse({'success': True, 'id': recommendation.id, 'title': recommendation.title, 'description': recommendation.description})
#     return JsonResponse({'success': False}, status=400)

# @csrf_exempt
# def recommendation_update(request, recommendation_id):
#     """Actualizar una recomendación existente."""
#     recommendation = get_object_or_404(Recommendation, id=recommendation_id)
#     if request.method == 'POST':
#         recommendation.title = request.POST.get('title', recommendation.title)
#         recommendation.description = request.POST.get('description', recommendation.description)
#         recommendation.save()
#         return JsonResponse({'success': True, 'id': recommendation.id, 'title': recommendation.title, 'description': recommendation.description})
#     return JsonResponse({'success': False}, status=400)

# @csrf_exempt
# def recommendation_delete(request, recommendation_id):
#     """Eliminar una recomendación."""
#     recommendation = get_object_or_404(Recommendation, id=recommendation_id)
#     if request.method == 'POST':
#         recommendation.delete()
#         return JsonResponse({'success': True})
#     return JsonResponse({'success': False}, status=400)


# def recommendations_dashboard(request):
#     agronomist_id = request.session.get('agronomist_id')
#     if not agronomist_id:
#         raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

#     """
#     Vista para el tablero de recomendaciones personalizadas.
#     """
#     crops = Crop.objects.all()  # Obtenemos todos los cultivos.
#     has_crops = crops.exists()

#     return render(request, 'admin/recommendation_list.html', {
#         'crops': crops,
#         'has_crops': has_crops
#     })


# def crop_data_for_recommendations(request):
#     """
#     Endpoint para devolver datos de cultivos para poblar gráficos y la tabla.
#     """
#     crops = Crop.objects.all()

#     crop_data = []
#     for crop in crops:
#         observations = crop.observations.order_by('-date')  # Observaciones más recientes
#         last_observation = observations.first() if observations.exists() else None

#         crop_data.append({
#             'id': crop.id,
#             'name': crop.name,
#             'field': crop.field.name,
#             'type': crop.get_crop_type_display(),
#             'expected_yield': crop.yield_estimate or 0,
#             'last_observation': last_observation.date if last_observation else None,
#             'status': 'Alerta' if observations.filter(notes__icontains="problema").exists() else 'Estable'
#         })

#     return JsonResponse({'crops': crop_data})


# def create_recommendation(request):
#     """
#     Crear una recomendación personalizada para un cultivo.
#     """
#     if request.method == 'POST':
#         crop_id = request.POST.get('crop_id')
#         title = request.POST.get('title')
#         description = request.POST.get('description')
#         priority = request.POST.get('priority')

#         crop = get_object_or_404(Crop, id=crop_id)
#         Recommendation.objects.create(
#             crop=crop,
#             title=title,
#             description=description,
#             priority=priority
#         )

#         # Redirigir al tablero de recomendaciones
#         return JsonResponse({'success': True})

#     return JsonResponse({'error': 'Método no permitido'}, status=405)


@user_required('agronomist')
def general_reports_list(request):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

    """
    Renderiza la lista de reportes generales disponibles para el agrónomo.
    """
    reports = [
        {
            "title": _("Reporte General de Cultivos"),
            "description": _("Incluye información detallada sobre todos los cultivos en la plataforma."),
            "url": "general-crop-report",
        },
        {
            "title": _("Reporte General de Scouts"),
            "description": _("Incluye información sobre todos los scouts registrados en la plataforma."),
            "url": "general-scout-report",
        },
        {
            "title": _("Reporte General de Maquinaria"),
            "description": _("Incluye información detallada sobre toda la maquinaria registrada y su estado."),
            "url": "general-machinery-report",
        },
        {
            "title": _("Reporte de Problemas Detectados"),
            "description": _("Listado de plagas, deficiencias y otros problemas detectados en las observaciones de cultivos."),
            "url": "general-problems-report",
        },
    ]

    return render(request, 'admin/report_list.html', {'reports': reports})


@user_required('agronomist')
def general_crop_report(request):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

    """
    Genera un reporte general de los cultivos en la plataforma.
    """
    # Crear el archivo Excel
    wb = Workbook()
    ws = wb.active
    ws.title = _("Reporte General de Cultivos")

    # Encabezados
    ws.append([
        _("ID del Cultivo"),
        _("Nombre del Cultivo"),
        _("Tipo"),
        _("Campo"),
        _("Scout Asignado"),
        _("Finquero"),
        _("Número de Plantas"),
        _("Rendimiento Estimado"),
        _("Precio de Mercado"),
        _("Ingreso Estimado"),
    ])

    # Datos
    crops = Crop.objects.select_related('field__scout__farmer').all()
    for crop in crops:
        scout = crop.field.scout
        farmer = scout.farmer if scout else None
        num_plants = crop.num_plants or 0
        yield_per_plant = crop.yield_per_plant or 0
        market_price = crop.market_price or 0
        estimated_income = num_plants * yield_per_plant * market_price
        ws.append([
            crop.id,
            crop.name,
            crop.get_crop_type_display(),
            crop.field.name,
            f"{scout.first_name} {scout.last_name}" if scout else _(
                "No asignado"),
            f"{farmer.first_name} {farmer.last_name}" if farmer else _(
                "No asignado"),
            num_plants,
            yield_per_plant,
            market_price,
            estimated_income,
        ])

    # Descargar el archivo
    response = HttpResponse(
        content_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
    )
    response["Content-Disposition"] = f'attachment; filename="general_crop_report.xlsx"'
    wb.save(response)
    return response


@user_required('agronomist')
def general_scout_report(request):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

    """
    Genera un reporte general de los Scouts en la plataforma.
    """
    # Crear el archivo Excel
    wb = Workbook()
    ws = wb.active
    ws.title = _("Reporte General de Scouts")

    # Encabezados
    ws.append([
        _("ID del Scout"),
        _("Nombre Completo"),
        _("Email"),
        _("Teléfono"),
        _("Finquero Asignado"),
        _("Número de Campos"),
        _("Número de Cultivos"),
    ])

    # Datos
    scouts = Scout.objects.prefetch_related(
        'fields_managed', 'fields_managed__crops', 'farmer').all()
    for scout in scouts:
        farmer = scout.farmer
        fields_count = scout.fields_managed.count()
        crops_count = Crop.objects.filter(
            field__scout=scout).count()  # Conteo correcto

        ws.append([
            scout.id,
            f"{scout.first_name} {scout.last_name}",
            scout.email,
            scout.phone or _("No disponible"),
            f"{farmer.first_name} {farmer.last_name}" if farmer else _(
                "No asignado"),
            fields_count,
            crops_count,
        ])

    # Descargar el archivo
    response = HttpResponse(
        content_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
    )
    response["Content-Disposition"] = f'attachment; filename="general_scout_report.xlsx"'
    wb.save(response)
    return response


@user_required('agronomist')
def edit_agronomist_profile(request):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        raise ValueError(_("El ID del Agrónomo no se encuentra en la sesión"))

    agronomist = get_object_or_404(Agronomist, id=agronomist_id)

    if request.method == 'POST':
        form = AgronomistProfileForm(
            request.POST, request.FILES, instance=agronomist)
        if form.is_valid():
            form.save()
            messages.success(request, _(
                "Tu perfil ha sido actualizado exitosamente."))
            return redirect('edit-agronomist-profile')
        else:
            messages.error(request, _(
                "Hubo un error al actualizar tu perfil. Por favor, revisa los datos."))
    else:
        form = AgronomistProfileForm(instance=agronomist)

    return render(request, 'admin/edit_agronomist_profile.html', {'form': form})



#SOCIAL NETWORK

@user_required('agronomist')
def post_list(request):
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        return redirect('login')  # Redirigir al login si no está autenticado

    # Obtener el ContentType para Agronomist
    agronomist_content_type = ContentType.objects.get_for_model(Agronomist)

    # Obtener los posts del usuario actual
    posts = Post.objects.all().order_by('-created_at').prefetch_related('comments')

    # Pasar el ContentType y el ID del usuario a la plantilla
    return render(request, 'admin/post_list.html', {
        'posts': posts,
        'agronomist_content_type': agronomist_content_type,
        'agronomist_id': agronomist_id,
    })

@user_required('agronomist')
def create_post(request):
    agronomist_id = request.session.get('agronomist_id')
    agronomist_content_type = ContentType.objects.get_for_model(Agronomist)

    if request.method == 'POST':
        form = PostForm(request.POST)
        if form.is_valid():
            post = form.save(commit=False)
            post.author_type = agronomist_content_type
            post.author_id = agronomist_id
            post.save()  # Guardar el post en la base de datos
            return redirect('post_list')
    else:
        form = PostForm()

    posts = Post.objects.filter(author_type=agronomist_content_type, author_id=agronomist_id)
    return render(request, 'admin/post_list.html', {'form': form, 'posts': posts})


@user_required('agronomist')
def edit_post(request, post_id):
    agronomist_id = request.session.get('agronomist_id')
    agronomist_content_type = ContentType.objects.get_for_model(Agronomist)

    # Obtener el post solo si el autor es el usuario actual
    post = get_object_or_404(Post, id=post_id, author_type=agronomist_content_type, author_id=agronomist_id)

    if request.method == 'POST':
        form = PostForm(request.POST, instance=post)
        if form.is_valid():
            form.save()
            return redirect('post_list')  # Redirigir a la lista de posts
    else:
        form = PostForm(instance=post)

    # Mostrar el formulario de edición en la misma página de lista de posts
    posts = Post.objects.filter(author_type=agronomist_content_type, author_id=agronomist_id)
    return render(request, 'admin/post_list.html', {'form': form, 'posts': posts, 'editing_post': post})    

@user_required('agronomist')  # Asegúrate de que el usuario esté autenticado como Agronomist
def delete_post(request, post_id):
    # Obtener el ID del usuario actual desde la sesión
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        messages.error(request, "No estás autenticado.")
        return redirect('login')  # Redirigir al login si no está autenticado

    # Obtener el ContentType para Agronomist
    agronomist_content_type = ContentType.objects.get_for_model(Agronomist)

    # Obtener el post solo si el autor es el usuario actual
    post = get_object_or_404(Post, id=post_id, author_type=agronomist_content_type, author_id=agronomist_id)

    if request.method == 'POST':
        # Eliminar el post
        post.delete()
        messages.success(request, "La publicación se ha eliminado correctamente.")
        return redirect('post_list')  # Redirigir a la lista de posts

    # Si no es una solicitud POST, redirigir a la lista de posts
    return redirect('post_list')

@user_required('agronomist')  # Asegúrate de que el usuario esté autenticado como Agronomist
def add_comment(request, post_id):
    # Obtener el ID del usuario actual desde la sesión
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        messages.error(request, "No estás autenticado.")
        return redirect('login')  # Redirigir al login si no está autenticado

    # Obtener el ContentType para Agronomist
    agronomist_content_type = ContentType.objects.get_for_model(Agronomist)

    # Obtener el post al que se agregará el comentario
    post = get_object_or_404(Post, id=post_id)

    if request.method == 'POST':
        form = CommentForm(request.POST)
        if form.is_valid():
            comment = form.save(commit=False)
            comment.post = post
            comment.author_type = agronomist_content_type
            comment.author_id = agronomist_id
            comment.save()
            messages.success(request, "El comentario se ha agregado correctamente.")
            return redirect('post_list')  # Redirigir a la lista de posts
    else:
        form = CommentForm()

    # Si no es una solicitud POST, redirigir a la lista de posts
    return redirect('post_list')


@user_required('agronomist')  # Asegúrate de que el usuario esté autenticado como Agronomist
def edit_comment(request, comment_id):
    # Obtener el ID del usuario actual desde la sesión
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        messages.error(request, "No estás autenticado.")
        return redirect('login')  # Redirigir al login si no está autenticado

    # Obtener el ContentType para Agronomist
    agronomist_content_type = ContentType.objects.get_for_model(Agronomist)

    # Obtener el comentario solo si el autor es el usuario actual
    comment = get_object_or_404(Comment, id=comment_id, author_type=agronomist_content_type, author_id=agronomist_id)

    if request.method == 'POST':
        form = CommentForm(request.POST, instance=comment)
        if form.is_valid():
            form.save()
            messages.success(request, "El comentario se ha editado correctamente.")
            return redirect('post_list')  # Redirigir a la lista de posts
    else:
        form = CommentForm(instance=comment)

    # Pasar el formulario a la plantilla
    posts = Post.objects.filter(author_type=agronomist_content_type, author_id=agronomist_id)
    return render(request, 'admin/post_list.html', {
        'form': form,
        'posts': posts,
        'editing_comment': comment,  # Para identificar que estamos editando un comentario
    })


@user_required('agronomist')  # Asegúrate de que el usuario esté autenticado como Agronomist
def delete_comment(request, comment_id):
    # Obtener el ID del usuario actual desde la sesión
    agronomist_id = request.session.get('agronomist_id')
    if not agronomist_id:
        messages.error(request, "No estás autenticado.")
        return redirect('login')  # Redirigir al login si no está autenticado

    # Obtener el ContentType para Agronomist
    agronomist_content_type = ContentType.objects.get_for_model(Agronomist)

    # Obtener el comentario solo si el autor es el usuario actual
    comment = get_object_or_404(Comment, id=comment_id, author_type=agronomist_content_type, author_id=agronomist_id)

    if request.method == 'POST':
        comment.delete()
        messages.success(request, "El comentario se ha eliminado correctamente.")
        return redirect('post_list')  # Redirigir a la lista de posts

    # Si no es una solicitud POST, redirigir a la lista de posts
    return redirect('post_list')



def resource_dashboard(request):
    books = Book.objects.all()
    videos = Video.objects.all()
    return render(request, 'admin/resources.html', {'books': books, 'videos': videos})

@csrf_exempt
def create_or_update_book(request):
    if request.method == 'POST':
        data = request.POST
        book_id = data.get('id')
        title = data.get('title')
        description = data.get('description')
        file = request.FILES.get('file')

        if book_id:
            book = get_object_or_404(Book, id=book_id)
            book.title = title
            book.description = description
            if file:
                book.file = file
            book.save()
        else:
            book = Book.objects.create(title=title, description=description, file=file)

        return JsonResponse({'success': True})

@csrf_exempt
def delete_book(request, pk):
    book = get_object_or_404(Book, pk=pk)
    book.delete()
    return JsonResponse({'success': True})

@csrf_exempt
def create_or_update_video(request):
    if request.method == 'POST':
        data = json.loads(request.body)
        video_id = data.get('id')
        title = data.get('title')
        description = data.get('description')
        youtube_url = data.get('youtube_url')

        if video_id:
            video = get_object_or_404(Video, id=video_id)
            video.title = title
            video.description = description
            video.youtube_url = youtube_url
            video.save()
        else:
            Video.objects.create(title=title, description=description, youtube_url=youtube_url)

        return JsonResponse({'success': True})

@csrf_exempt
def delete_video(request, pk):
    video = get_object_or_404(Video, pk=pk)
    video.delete()
    return JsonResponse({'success': True})            